#     Copyright 2007 Jim Bublitz <jbublitz@nwinternet.com>
#     Earlier copyrights 2001-6 Jim Bublitz may also apply
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

import sys, os, os.path, shutil, getopt
import cStringIO

# Handle the project file
from project import ProjectData, ModuleData

from toplevel import TopLevelFile
from parseTopLevel import TopLevelParser

# Datastructures passed between various modules
from modinfo import ModuleInfo, ImportModule

from twineConfig import buildFileSystem, docs

class PresipError (Exception):
    """
    The base class for presip errors
    """
    def __init__ (self, msg = ""):
        self.msg = msg

    def __str__ (self):
        return self.msg

def error (msg, caption = ""):
    sys.stderr.write ("\nError:   %s\n" % (msg))
    sys.exit (1)

def formTopLevelName (prj, mod_name, basename = False):
    if not basename:
        filename = os.path.join (prj.prj_destination, "sip", mod_name, mod_name) + "mod.sip.in"
    else:
        filename =  mod_name + "mod.sip"

    return filename

def copydir (dir, src, dst):
    if dir and dir [0] == ".":
        return

    path = os.path.join (src, dir)

    dstpath = os.path.join (dst, dir)
    if not os.path.exists (dstpath):
        os.mkdir (dstpath)

    dirlist = os.listdir (path)

    for item in dirlist:
        itempath = os.path.join (path, item)

        if os.path.isdir (itempath):
            copydir (item, path, dstpath)
        elif item [0] != ".":
            shutil.copyfile (itempath, os.path.join (dstpath, item))

def movePrevFiles (paths, dst):
    for path in paths:
        copydir ("", path, dst)

def adjustImports (prjdata, moddata, moduleList):
    if not moduleList:
        return

    for i in range (len (moddata.mod_import)):
        if not moddata.mod_import [i][0].endswith (".sip")\
            and moddata.mod_import [i][0] in moduleList:
            moddata.mod_import [i] = (formTopLevelName (prjdata, moddata.mod_import [i][0], True),\
                 moddata.mod_import [i][1], moddata.mod_import [i][2], moddata.mod_import [i][3],)

def applyDiff (prjdata, moddata):
    vers     = prjdata.prj_version.replace ("_", "").lower ()
    basename = os.path.join (prjdata.prj_destination, "sip", moddata.mod_name, moddata.mod_name)
    diffname = "".join ([basename, "-", vers, ".diff"])
    plus  = []
    minus = []
    if os.path.exists (diffname):
        d = open (diffname)
        for line in d:
            if line [2:].startswith ("%Include"):
                if line.startswith ("+"):
                    plus.append (line [2:])
                elif line.startswith ("-"):
                    minus.append (line [2:])
        d.close ()                    

    sipin  = open (formTopLevelName (prjdata, moddata.mod_name))
    sipout = open ("".join ([basename, "mod.sip"]), "w")

    for line in sipin:
        if line.startswith ("%Include"):
            inclFound = True
            if minus and line in minus:
                continue
            sipout.write (line)
        elif line.startswith ("@mark@"):
            for p in plus:
                sipout.write (p)
        else:
            sipout.write (line)

    sipin.close ()
    sipout.close ()

    return "".join ([basename, "mod.sip"])


def main (prjfile = None):
    # look for the project file
    if not prjfile:
        prjfile = sys.argv [1]

    if not prjfile:
        error ("No project file specified")

    # get the global project data
    prjfd   = open (prjfile, "r")
    prjData = ProjectData (prjfd)
    line    = prjData.readPrjData () # reads project level data and returns first 'module =' line

    # the buildFileSystem plugin identified in twineConfig
    buildFileSystem (prjfile, prjData) 
    
    if not line:
        error ("No module data in project file")

    if not line.startswith ("module"):
        error ("Expected module data but got: < %s >" % line)

    # get the data for the first module
    modData = ModuleData (prjfd, prjData)
    mi      = ModuleInfo ()

    mi.sipDst  = prjData.prj_sipDir

    moduleList = []

    # copy the old sip files - if a sip file doesn't exist
    # in the current version, we'll still need it to build
    # the previous version and this is the easiest way to
    # sort that out
    if prjData.prj_previous and prjData.prj_prevpath:
        movePrevFiles (prjData.prj_prevpath, os.path.join (mi.sipDst))

    # create the top level parser
    topLevelParser = TopLevelParser ()

    # loop through modules, starting with the one that's ready to go from above
    while not modData.eof and line and not line.startswith ("build") and not line.startswith ("doc"):

        # the first line of the next module block is the end condition
        # for this module block
        line = modData.readModData (line)
        print "\n==== module %s ====\n" % modData.mod_name

        adjustImports (prjData, modData, moduleList)

        if not modData.eof:
            # transfer some of the data to the mi structure
            mi.init (prjData, modData)
            
            moduleList.append (mi.modname)

            # transfer the import data
            impData = {}
            for tpl in modData.mod_import:
                imp = ImportModule ()
                imp.importFile, imp.versionLow, imp.versionHigh, imp.versPlatform = tpl
                imp.importFile = os.path.basename (imp.importFile)

                # expand local module names
                if imp.importFile in moduleList:
                    imp.importFile = formTopLevelName (prjData, imp.importFile, True)

                impData [imp.importFile] = imp

            # add the destination directory as an import search path
            if not mi.sipDst in mi.importPath:
                mi.importPath.append (mi.sipDst)

            #!!!Fixme - need a switch here; right now this prevents
            # edits to a toplevel file from being lost, but it also
            # prevents broken toplevel files from being replaced
            tlf = TopLevelFile (prjData, modData, mi.isKDE4)
            tlf.writeSipin()

            # create the top level sip file - toplevel is the path to the file
            toplevel = applyDiff (prjData, modData)
            print toplevel

            # read and parse the top level sip file
            topLevelParser.processModule (toplevel, mi)

            # this starts the whole process for the module - it reads in each h file (and
            # matching sip file if versioning), does the parsing and spits out the new
            # sip file

            if not modData.mod_retain:
                mi.clear ()
            else:
                # add previous module to list of imports retained
                topLevelParser.modules.append (mi.modname)
#                print topLevelParser.modules
                mi.reset ()

    prjfd.close ()
    allClassesWriter = docs (None, mi, None, project=prjData)
    allClassesWriter.writeAllClasses ()
    print

if __name__ == '__main__':
    try:
        optlist, args = getopt.getopt(sys.argv[1:],"d")
    except getopt.GetoptError:
        sys.exit (1)

    main (sys.argv [-1])
