#     Copyright 2006 Jim Bublitz <jbublitz@nwinternet.com>
#     Earlier copyrights 2001-5 Jim Bublitz may also apply
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

import os, time, sys
import cmake

copyright_string = """//
//     Copyright %i Jim Bublitz <jbublitz@nwinternet.com>
//     Earlier copyrights 1998 - %s Jim Bublitz also apply

"""

presip_header = """
//                 Generated by preSip
//            PyKDE4 module %s  version %s

"""


class TopLevelFile(object):
    def __init__ (self, prj, mod, isKDE4):
        self.prj         = prj
        self.mod         = mod
        self.mname       = self.mod.mod_name
        basename         = os.path.join (self.prj.prj_destination, "sip", self.mname, self.mname)
        self.sipin       = basename + "mod.sip.in"
        self.vers        = self.prj.prj_version.replace ("_", "").lower ()
        self.diff        = "".join ([basename, "-", self.vers, ".diff"])
        self.isKDE4      = isKDE4
        self.compareList = []

    def writeSipin(self):
        if os.path.exists (self.sipin):
            self.writeDiffFile ()
        else:
            self.writeSipinFile ()

    def writeDiffFile (self):
        si = open (self.sipin)
        files = []
        for line in si:
            if line.startswith ("%Include"):
                fn = line.split () [1].strip ()
                if "/" in fn:
                    fn = fn.split ("/") [0].strip ()
                files.append (fn)
        si.close ()
        
        srcList = [os.path.basename (f) for f in self.getFileList ()]

        isDiff = False
        df = open (self.diff, "w")
        for file in files:
            f = file.replace (".sip", ".h")
            if f not in srcList and not (file in self.mod.mod_require or file in self.mod.mod_noheader):
                df.write ("- %%Include %s\n" % file)
                isDiff = True

        for file in srcList:
            f = file.replace (".h", ".sip")
            if f not in files:
                df.write ("+ %%Include %s\n" % f)
                isDiff = True

        df.close ()
        if not isDiff:
            os.unlink (self.diff)

        if self.mod.mod_timeline:
            self.updateTimeline (self.mod.mod_name, self.mod.mod_timeline)


    def writeSipinFile (self):
        if not os.path.exists (os.path.dirname (self.sipin)):
            os.mkdir (os.path.dirname (self.sipin))
        self.fd = open (self.sipin, "w")

        self.writeCopyright (self.prj.prj_copyright)

        if self.prj.prj_license:
            self.writeLicense (self.prj.prj_license)

        self.writeModule (self.mod.mod_name)
        if self.mod.mod_timeline:
            self.writeTimeline (self.mod.mod_timeline)

        if self.mod.mod_platforms:
            self.writePlatforms (self.mod.mod_platforms)

        if self.mod.mod_import:
            self.writeImports (self.mod.mod_import)

#        self.writeExpDoc (self.mod.mod_name)

        self.writeIncludes ()
#        self.writeCloseDoc ()
#        self.writeCode (self.mod.mod_code)
#        self.writeMakefile ()

        self.fd.close ()

    def getFilename (self, version, modname):
        sfx = version.replace ("_", "")
        sfx = sfx.lower ()
        filename = os.path.join (self.prj.prj_destination, "sip", "-".join ([modname, sfx]) + ".sip")
        return filename

    def updateTimeline (self, modname, timeline):
            newFn = self.sipin + ".tmp"
            m = open (self.sipin, "r")
            n = open (newFn, "w")
            done = False
            for line in m:
                if done or line.find ("%Timeline") < 0:
                    n.write (line)
                else:
                    n.write ("%Timeline {")
                    for tl in timeline:
                        n.write (" %s " % tl)
                    n.write ("}\n")
                    done = True

            n.close ()
            m.close ()
            os.unlink (self.sipin)
            os.rename (newFn, self.sipin)

#    def getModulePath (self):
#        if not self.prj.prj_version:
#            filename = os.path.join (self.prj.prj_destination, "sip", self.mod.mod_name) + ".sip"
#        else:
#            filename = self.getFilename (self.prj.prj_version, self.mod.mod_name)

#        return filename

    def writeCopyright (self, cp):
        yr = time.localtime ()[0]
        if not cp:
            self.fd.write (copyright_string % (yr, yr - 1))

        elif cp.find ("%i") >= 0:
            self.fd.write (cp % yr)

        else:
            self.fd.write (cp)

        self.fd.write (presip_header % (self.mod.mod_name, self.prj.prj_version))

    def writeLicense (self, lic):
        self.fd.write ("%Copying\n")
        self.fd.write (lic.replace ("//", ""))
        self.fd.write ("%End\n")

    def writeModule (self, module):        
        if self.prj.prj_package:
            fullname = ".".join ([self.prj.prj_package, module])
        else:
            fullname = module            
                   
        self.fd.write ("\n%%Module %s\n\n" % fullname)

    def writeTimeline (self, tline):
        self.fd.write ("%Timeline {")
        for tl in tline:
            self.fd.write (" %s " % tl)
        self.fd.write ("}\n\n")

    def writePlatforms (self, plat):
        self.fd.write ("%Platforms {")
        for p in plat:
            self.fd.write (" %s " % p)
        self.fd.write ("}\n\n")

    def writeImports (self, imps):
        for imp in imps:
            if imp [0].endswith ("mod.sip"):
                self.fd.write ("%%Import %s\n" % imp [0])
            else:
                self.fd.write ("%%Import %smod.sip\n" % imp [0])
        self.fd.write ("\n")

#    def writeExpDoc (self, name):
#        self.fd.write ("%ExportedDoc\n</Sect1>\n<Sect1>\n<Title>\n")
#        self.fd.write ("<Literal> %s </Literal> Module Reference\n</Title>\n%%End\n\n" % name)

    def createCompareList (self):
        okList = self.mod.mod_compare + self.mod.mod_subst

        for dir in okList:
            if os.path.isdir (dir):
                files = os.listdir (dir)
                for file in files:
                    if file [-2:] == ".h" and os.path.isfile (os.path.join (dir, file)):
                        self.compareList.append (file)
        
        if self.prj.prj_complist:
            match0 = self.prj.prj_complist [0]
            match1 = "_".join ([self.mname, "HEADERS"])
            for dir in self.mod.mod_source:
                matchfile = os.path.join (dir, self.prj.prj_complist [1])
                if os.path.exists (matchfile):
                    m = open (matchfile, "r")
                    fullLine = ""
                    for line in m:
                        ln = line.strip ()
                        if ln.endswith ("\\"):
                            fullLine += ln.replace ("\\", " ")
                            continue
                        else:
                            fullLine += ln

                        if "=" in fullLine and (match0 in fullLine or match1 in fullLine):
                            flist0 = fullLine.split ("=") [1].strip ()
                            flist  = flist0.split ()

                            for file in flist:
                                if file [-2:] == ".h" and not file in self.compareList:
                                    self.compareList.append (file)

                        fullLine = ""
                    m.close ()
                    
    def filt (self, f):
        if f in self.mod.mod_ignore:
            return False

        return f in self.compareList or (not self.mod.mod_compare and not self.mod.mod_subst and not self.prj.prj_complist)

    def getFileList (self,absolute=False):
        if self.isKDE4:
            return self.getFileList4 (absolute)
        
        self.createCompareList ()
        srcList  = []

        for dir in self.mod.mod_source:
            tmpList  = filter (lambda f: os.path.isfile (os.path.join (dir, f)), os.listdir (dir))
            srcList += filter (self.filt, tmpList)

        srcList.sort ()
        return srcList

    def getFileList4 (self,absolute=False):
        srcList = []
        count = 0
        for dir in self.mod.mod_source:
            cmakefile = os.path.join (dir, "CMakeLists.txt")
            if not os.path.exists (cmakefile):
                continue
                #print "Can't find path %s - terminating (toplevel.py:getFileList4())" % cmakefile
                #sys.exit (-1)
            print("Extracting .h file list from " + cmakefile)
            new_list = ExtractInstallFiles(cmakefile)
            for filename in new_list:
                if not os.path.split (filename)[1] in self.mod.mod_ignore:
                    if absolute:
                        srcList.append (os.path.join(dir,filename))
                    else:
                        srcList.append (filename)
                    count += 1 
        print("Found %i .h files." % count)
        srcList.sort ()
#        print(srcList)
        return srcList
        

    def writeIncludes (self):
        for f in self.mod.mod_require + self.mod.mod_noheader:
            self.fd.write ("%%Include %s\n" % f)

        srcList = self.getFileList ()
        for fh in srcList:
            fsip = ".".join ([os.path.splitext (os.path.split(fh)[1])[0], "sip"])
            self.fd.write ("%%Include %s\n" % fsip)

        self.fd.write ("\n")

def ExtractInstallFiles(cmakefile):
    return cmake.ExtractInstallFiles(cmakefile)

def oldExtractInstallFiles(cmakefile):
    STATE_START = 0
    STATE_INSTALL = 1
    STATE_INSTALL_FILES =2
        
    m = open (cmakefile, "r")
    count = 0
    srcList = []
    state = STATE_START
    for line in m:
        line = line.strip()
        if state == STATE_START:
            if line.startswith("install") or line.startswith("INSTALL"):
                state = STATE_INSTALL
                line = line [7:]
                line = line.strip()
                if line == "" or line == "(":
                    continue

        if state == STATE_INSTALL:
            pos  = line.find ("FILES")
            if pos < 0:
                state = STATE_START
                continue
            
            if pos + 5 > len (line):
                continue
            
            line = line [pos + 5:]
            line  = line.strip ()
            state = STATE_INSTALL_FILES
        
        if state == STATE_INSTALL_FILES:
            items = line.split ()
           
            if items:
                for item in items:
                    if '#' in item:
                        break
                        
                    if ("$" in item and not item.endswith(".h")) or "DESTINATION" in item or ")" in item:
                        state = STATE_START
                        continue

                    if not item.endswith (".h") or "$" in item:
                        continue

                    srcList.append(item)
    m.close ()
    return srcList
    
#    def writeCloseDoc (self):
#        self.fd.write ("%ExportedDoc\n</Sect1>\n%End\n\n")

#    def writeCode (self, code):
#        for line in self.mod.mod_code:
#            self.fd.write (line)

#    def writeMakefile (self):
#        self.fd.write ("%%Makefile %s.pro\n" % self.mod.mod_name)

#        sfx = self.prj.prj_version.replace ("_", "")
#        sfx = sfx.lower ()

#        yr = time.localtime ()[0]
#        cp = self.prj.prj_copyright
#        if not cp:
#            cp = doc.copyright_string % (yr, yr - 1)

#        elif cp.find ("%i") >= 0:
#            cp =  cp % yr

#        self.fd.write (cp.replace ("//", "#"))
#        psHdr = doc.presip_header % (self.mod.mod_name, self.prj.prj_version)
#        self.fd.write (psHdr.replace ("//", "#"))

#        self.fd.write ("TEMPLATE = lib\n")
#        self.fd.write ("TARGET = @BL_TARGET@\n")
#        self.fd.write ("DESTDIR = @BL_DESTDIR@\n")
#        self.fd.write ("CONFIG += qt warn_off release dll @BL_THREAD@\n")
#        self.fd.write ("INCLUDEPATH = %s @BL_INCLUDEPATH@ %s\n" % (os.path.join ("..", "extra", sfx), " #".join (self.mod.mod_include)))
#        self.fd.write ("DEFINES = SIP_MAKE_MODULE_DLL @BL_DEFINES@\n")
#        self.fd.write ("unix:LIBS += -L@BL_LINKDIRS@ %s\n" % " ".join (self.mod.mod_libs))
#        self.fd.write ("SOURCES = \\\n")
#        self.fd.write ("\t$B\n")
#        self.fd.write ("HEADERS = \\\n")
#        self.fd.write ("\t$H\n")
#        self.fd.write ("%End\n\n")
