#     Copyright 2006 Jim Bublitz <jbublitz@nwinternet.com>
#     Earlier copyrights 2001-5 Jim Bublitz may also apply
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

import sys, os, os.path, string, time, re

from modinfo import ModuleInfo

prjdata     = ["project", "destination", "version", "secondary", "previous", "prevpath", "license", "copyright",\
               "source", "complist", "compare", "importpath", "package", "define", "apidoxcommon", "apidoximage"]
moduleData  = ["module", "source", "ignore", "include", "compare", "import", "importpath", "libs", "code",\
               "subst", "retain", "timeline", "platforms", "amend", "sipslot", "define", "trace", "require",\
               "noheader", "cast", "subclass", "stop", "nobase", "nocast", "nonamespace", "subbase", "amendTypes", "maindox"]
buildData   = ["release", "version", "usingQt", "usingKDE", "option", "make", "makefilegen"]
endData     = ["module", "build", "doc", "stop"]


def parseLine (prjfile, line):
    fullline = ""
    while line:
        if line.find ("//") >= 0 or not line.strip ():
            line = prjfile.readline ()
            continue

        fullline += line.strip ()
        if fullline [-1] == "\\":
            fullline = fullline.replace ("\\", " ")
            line = prjfile.readline ()
            continue

        plist = string.split(string.strip (fullline), "=")
        fullline = ""
        length = len (plist)
        if length > 0:
            directive = plist [0].strip ()

        if length == 2:
            params = plist [1].strip ()
        else:
            params = ""

        return directive, params, line

    return (None, None, line)

def error (msg):
    sys.stderr.write ("\nError:  %s\n\n" % msg)
    sys.exit (1)


class ProjectData:
    """
    Parses the project data section of a project file
    """
    def __init__ (self, prjfile):
        self.prjfile         = prjfile
        self.prj_project     = ""
        self.prj_destination = ""
        self.prj_sipDir      = ""
        self.prj_version     = ""
        self.prj_secondary   = []
#        self.prj_timeline    = ""
        self.prj_previous    = ""
        self.prj_prevpath    = []
        self.prj_license     = "" #doc.lgpl_license_string
        self.prj_copyright   = ""
        self.prj_source      = ""
        self.prj_complist    = []
        self.prj_compare     = ""
        self.prj_importpath  = ""
        self.prj_isKDE4      = False
        self.prj_package     = ""
        self.prj_defines     = []
        self.error           = error
        self.prj_apidoxcommon = None
        self.prj_apidoximage = None
        
    def readPrjData (self):
        line = self.prjfile.readline ()
        while line:
            directive, params, line = parseLine (self.prjfile, line)
            if directive == None:
                return None

            if directive in endData:
                self.validate ()
                return line

            if directive in prjdata:
                # jump to the method pointed to by the directive's name
                exec ("".join (["self.", directive, " ('", params, "')"]))

            else:
                self.error ("Unknown directive '%s' in project data section\n" % directive)

            line = self.prjfile.readline ()

    # directives either set a flag, or set a value/list of values
    # the order in which directives appear should be irrelevant
    # (except for the 'end' directive)
    def project (self, params):
        self.prj_project = params
        if params == "PyKDE4":
            self.prj_isKDE4 = True

    def package (self, params):
        self.prj_package = params
    
    def destination (self, params):
        if params:
            self.prj_destination = params
            self.prj_sipDir = os.path.join (self.prj_destination, "sip")
        else:
            self.prj_destination = ""
            return

        try:
            if not os.path.exists (self.prj_destination):
                os.mkdir (self.prj_destination)
            if not os.path.exists (self.prj_sipDir):
                os.mkdir (self.prj_sipDir)
        except:
            self.error ("Failed to create destination directories")

    def source (self, params):
        if params:
            self.prj_source = params

        if not os.path.exists (self.prj_source):
            self.error ("Global project source path (source =) does not exist")

    def complist (self, params):
        if params:
            self.prj_complist = params.split ()

        if self.prj_complist != [] and len (self.prj_complist) != 2:
            self.error ("Global complist statement has wrong number of arguments")

    def compare (self, params):
        if params:
            self.prj_compare = params

        if params and not os.path.exists (self.prj_compare):
            self.error ("Global comparison source path (compare =) does not exist")

    def importpath (self, params):
        if params:
            self.prj_importpath = params

        if not os.path.exists (self.prj_importpath):
            self.error ("Global import path (importpath =) does not exist")

    def version (self, params):
        self.prj_version = params

    def secondary (self, params):
        self.prj_secondary += params.split ()

    def previous (self, params):
        self.prj_previous = params

    def prevpath (self, params):
        self.prj_prevpath.append (params)

    def license (self, params):
        pass
#        if not params in doc.licenses:
#            self.error ("Unknown license specified: %s\n" % (params))

#        self.prj_license = doc.licenses [params]

    def copyright (self, params):
        if params:
            self.prj_copyright = params

    def validate (self):
        if self.prj_destination == "" or self.prj_sipDir == "":
            self.error ("Destination path not specified")

        if self.prj_prevpath and not self.prj_previous:
            self.error ("Path to previous sip files specified, but no version given (previous =)")

        for path in self.prj_prevpath:
            if not os.path.exists (path):
                self.error ("Path to previous sip files %s does not exist" % path)

    def define (self, params):
        if not params:
            return

        try:
            x, y = eval (params)
        except ValueError:
            try:
                x, y = eval (params), ""
            except:
                self.error ("syntax error in 'define = %s' (missing quotes?)" % params)
        except:
            self.error ("syntax error in 'define = %s' (missing quotes?)" % params)

        self.prj_defines.append ((x, y))

    def apidoxcommon(self, params):
        src_re  = re.compile("\$SRC")
        self.prj_apidoxcommon = src_re.sub (self.prj_source, params)
        
    def apidoximage(self, params):
        src_re  = re.compile("\$SRC")
        self.prj_apidoximage = src_re.sub (self.prj_source, params)
        
class ModuleData:
    """
    Parses the module data section of a project file
    """
    def __init__ (self, prjfile, prjdata):
        self.prjfile = prjfile
        self.prjdata = prjdata
        self.eof     = False
        self.prjsrc  = prjdata.prj_source
        self.prjdst  = prjdata.prj_destination
        self.prjcmp  = prjdata.prj_compare
        self.prjimp  = prjdata.prj_importpath
        self.src_re  = re.compile ("\$SRC")
        self.dst_re  = re.compile ("\$DST")
        self.cmp_re  = re.compile ("\$COMP")
        self.imp_re  = re.compile ("\$IMP")
        self.error   = error
        self.prjmaindox = None

    def readModData (self, line):
        if self.eof or not line:
            self.eof = True
            return None

        self.mod_source      = []
        self.mod_ignore      = []
        self.mod_include     = []
        self.mod_compare     = []
        self.mod_import      = []
        self.mod_importpath  = ["", "./"]
        self.mod_libs        = []
        self.mod_code        = []
        self.mod_name        = ""
        self.mod_subst       = []
        self.mod_retain      = False
        self.mod_amendTypes  = []
        self.mod_amends      = []
        self.mod_sipslot     = []
        self.mod_defines     = []
        self.mod_fileDefines = {}
        self.mod_timeline    = []
        self.mod_platforms   = []
        self.mod_trace       = ""
        self.mod_require     = []
        self.mod_noheader    = []
        self.mod_cast        = []
        self.mod_nocast      = []
        self.mod_nonamespace = {}
        self.mod_subclass    = {}
        self.mod_nobase      = []
        self.mod_subbase     = {}
        self.accumCode       = False

        while line:
            if self.accumCode:
                self.code (line)

            else:
                directive, params, line = parseLine (self.prjfile, line)
                if directive == None:
                    self.validate ()
                    return None

                elif self.mod_name and directive in endData:
                    self.validate ()
                    return line

                if directive in moduleData:
                    if directive == "import": directive = "ximport"
                    exec ("".join (["self.", directive, " ('", self.expand (directive, params), "')"]))

                else:
                    self.error ("unknown directive '%s' in module data section\n" % directive)

            line = self.prjfile.readline ()

        return None

    def expand (self, directive, s):
        # expand $SRC, $DST macros
        s = self.src_re.sub (self.prjsrc, s)
        s = self.dst_re.sub (self.prjdst, s)
        s = self.cmp_re.sub (self.prjcmp, s)
        s = self.imp_re.sub (self.prjimp, s)

        if directive in ["define", "subclass", "maindox"]:
            return s
        else:
            # delimiter is ws - ignore any commas
            return s.replace (",", " ")

    def module (self, params):
        if not params:
            self.error ("No module name specified")

        self.mod_name = os.path.splitext (params)[0]

    def source (self, params):
        paths = params.split ()
        hPath = []
        for path in paths:
            hPath.append (path)
            subs = os.listdir (path)
            for sub in subs:
                if os.path.isdir (os.path.join (path, sub)):
                    hPath.append (os.path.join (path, sub))
      
        self.mod_source += hPath

    def ignore (self, params):
        self.mod_ignore += params.split ()

    def include (self, params):
        self.mod_include += params.split ()

    def compare (self, params):
        self.mod_compare += params.split ()

    def define (self, params):
        if not params:
            return

        if params [0].startswith ('"'):
            dst = 'module'
        elif params [0] == 'module':
            dst    = 'module'
            params = params [1:]
        else:
            dst    = 'file'
            file   = params [0]
            params = params [1:]
           
        try:
            x, y = eval (params)
        except ValueError:
            try:
                x, y = eval (params), ""
            except:
                self.error ("syntax error in 'define = %s' (missing quotes?)" % params)
        except:
            self.error ("syntax error in 'define = %s' (missing quotes?)" % params)

        if dst == 'module':
            self.mod_defines.append ((x, y))
        else:
            self.mod_fileDefines [file] = (x, y)

    def ximport (self, params):
        if not params:
            return
        param = params.split ("(")
        for i in range (len (param)):
            param [i] = param [i].strip ()

        if len (param) == 1:
            self.mod_import.append ((param [0], "-", "~", ""))
        else:
            if param [1][-1] != ")":
                self.error ("missing ')' - %s" % params)

            param [1] = param [1][:-1]
            vers = param [1].split ()
            for i in range (len (vers)):
                vers [i] = vers [i].strip ()

            if len (vers) == 2:
                self.mod_import.append ((param [0], vers [0], vers [1], ""))
            elif len (vers) == 3:
                self.mod_import.append ((param [0], vers [0], vers [1], vers [2]))
            else:
                self.error ("ill-formed version clause %s" % params)

    def importpath (self, params):
        self.mod_importpath += params.split ()

    def libs (self, params):
        self.mod_libs += params.split ()

    def require (self, params):
        self.mod_require += params.split ()

    def noheader (self, params):
        self.mod_noheader += params.split ()
    
    def code (self, params):
        if not self.accumCode:
            self.accumCode = True
        elif params.strip () != "end":
            self.mod_code.append (params)
        else:
            self.accumCode = False

    def subst (self, params):
        self.mod_subst += params.split ()

    def cast (self, params):
        self.mod_cast += params.split ()
    
    def nocast (self, params):
        self.mod_nocast += params.split ()

    def nonamespace (self, params):
        nsparams = params.split ()
        if len (nsparams) < 2:
            return
        
        file = nsparams [0].strip ()
        self.mod_nonamespace [file] = []
        for ns in nsparams [1:]:
            self.mod_nonamespace [file].append (ns.strip ())                         
    
    def subbase (self, params):
        subbaseparams = params.split ()
        if len (subbaseparams) < 2:
            return
        
        oldbase = subbaseparams [0].strip ()
        self.mod_subbase [oldbase] = subbaseparams [1].strip ()
    
    def subclass (self, params):
        aparams = params.split (",")
        cls = aparams [0].strip ()

        bases = []
        for b in aparams [2].split ():
            bases.append (b.strip ())
            
        self.mod_subclass [cls] = (aparams [1].strip (), bases)
    
    def retain (self, params):
        self.mod_retain = True

    def timeline (self, params):
        self.mod_timeline = params.split ()

    def platforms (self, params):
        self.mod_platforms = params.split ()

    def amendTypes (self, params):
        if not params:
            return

        atypes = params.split ()
        for atype in atypes:
            self.mod_amendTypes.append (atype.strip ())
            
    def amend (self, params):
        if not params:
            return
        aparams = params.split ()
        # aparams [0] is scope
        # aparams [1] is function type
        if aparams [1] not in ["ctor", "dtor", "all"]:
            self.error ("error in amend type - ctor, dtor or all, not %s" % aparams [1])
            return
        
        # aparams [2] is variable name (eg - 'parent')
        # aparams [3] is annotation
        if not aparams [3] in ["TransferThis", "Transfer", "TransferBack"]:
            self.error ("not a sip directive - %s" % aparams [3])

        self.mod_amends.append (aparams)

    def sipslot (self, params):
        if not params:
            return
        param = params.split ()
        if len (param) < 3:
            self.error ("Sipslot specification is invalid (object method [classes])")

        # Object* variable name
        obj  = param [0].strip ()
        # char* variable name
        slot = param [1].strip ()
        
        # a list of scopes where the modifications apply
        mods = []
        for i in range (2, len (param)):
            mods.append (param [i].strip ())

        self.mod_sipslot.append ((obj, slot, mods))

    def trace (self, params):
        self.mod_trace = params

    def nobase (self, params):
        self.mod_nobase += params.split ()
    
    def validate (self):
        if not self.mod_source:
            self.error ("No source files specified")

        for path in self.mod_source:
            if not os.path.exists (path):
                self.error ("Source path %s does not exist" % (path))

        if self.mod_compare and not self.prjdata.prj_isKDE4:
            for path in self.mod_compare:
                if not os.path.exists (path):
                    self.error ("Compare path %s does not exist" % (path))

        if self.mod_importpath:
            for path in self.mod_importpath:
                if path and not os.path.exists (path):
                    self.error ("Import path %s does not exist" % (path))

        if self.mod_subst:
            for path in self.mod_subst:
                if not os.path.exists (path):
                    self.error ("Substitution path %s does not exist" % (path))

    def stop (self, params):
        pass

    def maindox(self, params):
        self.prjmaindox = params
        
class BuildData:
    def __init__ (self, prjfile):
        self.prjfile      = prjfile
        self.endOfBldData = False
        self.eof          = False
        self.readBuildData ()

    def readBuildData (self, line):
        self.bld_release          = ""
        self.bld_version          = ""
        self.bld_usingQt          = False
        self.bld_usingKDE         = False
        self.bld_userShortOptions = ""
        self.bld_userLongOptions  = []
        self.bld_userShortOpts    = {}
        self.bld_userOpts         = {}
        self.bld_userDescr        = {}
        self.bld_make             = False
        self.bld_makefilegen      = ""

        fullline = ""
        while line:
            directive, params, line = parseLine (self.prjfile, line)
            if directive in buildData:
                exec ("".join (["self.", directive, " ('", params, "')"]))
                if self.endOfBldData:
                    return
            else:
                sys.stderr.write ("unknown directive '%s' in build data section\n" % directive)
                sys.exit (-1)

            line = self.prjfile.readline ()

        self.eof = True

    def release (self, params):
        self.bld_release = params

    def version (self, params):
        self.bld_version = params

    def usingQt (self, params):
        self.bld_usingQt = True

    def usingKDE (self, params):
        self.bld_usingKDE = True

    def option (self, params):
        param = params.split ("(")
        if len (param) != 4:
            sys.stderr.write ("incorrect option clause: %s" % params)
            sys.exit (-1)

        opts  = param [0].split ()
        short = opts [0].strip ()
        if len (short) > 2 or short [0] != '-' or short [1] not in string.ascii_uppercase:
            sys.stderr.write ("short options must be a single uppercase letter: %s" % short)
            sys.exit (-1)

        long  = opts [0].strip ()
        if long [0:2] != "--":
            sys.stderr.write ("long options must begin with '--': %s" % long)
            sys.exit (-1)

        self.bld_userShortOptions += short [1]
        self.bld_userLongOptions.append (long [2:])
        self.bld_userShortOpts [short] = long
        args = param [1].split (")")
        if len (args) != 2:
            sys.stderr.write ("incorrect option clause: %s" % params)
            sys.exit (-1)

        self.bld_userOpts [long] = args [0].strip ()
        descr = args [1].strip ()
        argmodifier = ""
        for i in range (len (descr)):
            if descr [i] not in ["'", '"']:
                argmodifier += descr [i]
            else:
                descr    = descr [i:]
                break

        argmodifier = argmodifier.strip ()
        if modifier:
            self.bld_userShortOptions += ":"
            self.bld_userLongOptions [-1] += "="

        self.bld_userDescr [long] = (modifier, descr)

    def make (self, params):
        self.bld_make = True

    def makefilegen (self, params):
        self.bld_makefilegen = params

    def end (self, params):
        self.endOfBldData = True


def main (prjfile = None):
    if not prjfile:
        prjfile = sys.argv [1]

    if not prjfile:
        sys.stderr.write ("Error - no project file specified\n")
        sys.exit (-1)

    sys.stdout.write ("pscreate: creating sip-in files from project file %s\n" % prjfile)
    prjfd = open (prjfile, "r")
    projectData = ProjectData (prjfd)

    modData = ModuleData (prjfd)
    prjfd.close ()

    while not modData.eof:
        modData.readModData ()
        if not modData.eof:
            WriteSipinFile (projectData, modData)

if __name__ == '__main__':
    main ()
