#     Copyright 2007-8 Jim Bublitz <jbublitz@nwinternet.com>
#     Copyright 2008   Simon Edwards <simon@simonzone.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

import os.path, datetime
from plugins.writedocs import DocFileWriter

spaces = '                                                '

def writeSipFiles (symbolData, modData, stateInfo):
    sipFileWriter = SipFileWriter (symbolData, modData, stateInfo)
    sipFileWriter.write ()
    
class SipFileWriter(object):
    # this class writes the actual new sip files to disk
    def __init__ (self, symbolData, modData, stateInfo):
        self.symbolData = symbolData
        self.modData    = modData
        self.indentCol  = 0
        self.nameIndent = 24
        self.stateInfo  = stateInfo
        self.fileout    = None
        self.access     = None
        self.force      = False
        self.skipClass  = 0
        self.docWrite   = DocFileWriter (symbolData, modData, stateInfo)

        self.versionStack = []
    
    def indent (self, n = 0):
        if not self.indentCol:
            return ''
        else:
            return spaces [:self.indentCol + n]
            
    # the space between the return type and function name
    def gap (self, sofar):
        used = len (sofar)
        if used >= self.nameIndent - 1:
            return "  "
        else:
            return spaces [:self.nameIndent - used]
            
    def clearVersion (self, fileLines, obj = None):
        if obj and obj.objectType in ['endclass', 'endnamespace']:
            objName = obj.name
        else:
            objName = ''
            
        while self.versionStack and (not obj or self.versionStack [-1][2] == objName):
            self.versionStack.pop ()
            fileLines.append ('%End\n')
        
    def checkVersion (self, obj):
        if self.force:
            return             
            
        if obj.objectType in ['namespace', 'endnamespace']:
            if obj.skip:
                return ''
                
        if obj.objectType in ['class', 'namespace']:
            lock = obj.name
        else:
            lock = ''
                
        low = high = ''
        if obj.version.versionLow:
            low  = obj.version.versionLow
        if obj.version.versionHigh:
            high = obj.version.versionHigh            
        
        if not self.versionStack:
            if low or high:
                self.versionStack.append ((low, high, lock))
                return "\n%%If ( %s - %s )" % (low, high)
            
        else:
            if not low and not high:
                self.versionStack.pop ()
                return '%End\n'
                
            if low == self.versionStack [-1][0] and high == self.versionStack [-1][1]:
                return ''
                
            if low > self.versionStack [-1][0] and high < self.versionStack [-1][1]:
                self.versionStack.append ((low, high, lock))
                return "\n%%If ( %s - %s )" % (low, high)
        
            if low < self.versionStack [-1][0] or high > self.versionStack [-1][1]:
                self.versionStack.pop ()
                self.versionStack.append ((low, high, lock))
                return "%%End\n\n%%If ( %s - %s )" % (low, high)
        
    def write (self):   
        # loop through all of the files for this module
        for file in self.symbolData.files:
            self.skipClass  = 0
            self.skipNS     = 0
            self.filename = '%s.sip' % file
            self.createFile ()
            self.indentCol  = 0
            self.nameIndent = 24
            fileLines = []
            print("Writing %s" % self.filename)
            
            for obj in self.symbolData.fileData [file]:
                if self.skipClass:
                    if obj.objectType == 'class':
                        self.skipClass += 1
                    elif obj.objectType == 'endclass':
                        self.skipClass -= 1
                    continue
                elif self.skipNS:
                    if obj.objectType == 'endnamespace':
                        self.skipNS = 0
                    continue
                    
                if obj.objectType == 'namespace' and obj.ignore:
                    fileLines += self.format_namespace (obj)
                    self.skipNS = 1
                    continue
                                
                # //force ... //end blocks
                if self.forceChanged (obj):
                    if self.force:
                        self.clearVersion (fileLines, obj)
                        fileLines.append ('\n//force')
                    else:
                        fileLines.append ('//end\n')
                        
                #v = self.checkVersion (obj)
                #if v:
                #    fileLines.append (v)

                # access specifiers (public, protected, etc)
                changed, nextAccess = self.accessChanged (obj)                
                if changed:
                    fileLines.append ('\n%s%s:' % (self.indent (-4), self.access))
                self.access = nextAccess
                
                # choose the method that writes the object out and call it
                writer = eval ('self.format_' + obj.objectType)
                fileLines += writer (obj)

            # if the last object was //force'd, there won't be an
            # object with force == False to trigger the matching //end
            # - otherwise, make sure the file ends with a newline
            if self.force:
                fileLines.append ('//end\n')
            else:
                fileLines.append ('\n')
                
            self.clearVersion (fileLines)
            
            # finally write the file to disk
            if self.fileout:
                self.fileout.write ('\n'.join (fileLines))
                self.fileout.close ()           
        
        self.docWrite.writeMainIndex ()
        self.docWrite.writeNamespaces ()
        self.docWrite.writeClasses ()
        
    def createFile (self):
        dst = self.modData.sipDst

        file = os.path.join (dst, 'sip', self.modData.modname, self.filename)
        self.fileout = open (file, 'w')
        
        # Reuse the license header from the previous version of the sip file if
        # possible.
        old_sip_path = os.path.join(self.modData.prevpath[0],self.modData.modname, self.filename)
        if os.path.exists(old_sip_path):
            header = []
            old_sip = open(old_sip_path,'r')
            for line in old_sip.readlines():
                if line.strip()=="" or line.startswith("//"):
                    header.append(line)
                else:
                    break
            old_sip.close()

            self.fileout.write(''.join(header))

        else:
            # update the copyright dates and write the common header info
            yr = datetime.date.today ().year
            self.fileout.write (self.symbolData.copyright % (yr, yr - 1))
            version = self.modData.version.replace ('KDE_', 'KDE ')
            self.fileout.write (self.symbolData.header)
            self.fileout.write (self.symbolData.license)
    

          
    def formatTemplate (self, tmpl):         
        return ["\n%stemplate <%s>" % (self.indent (), tmpl)]        

    def formatArgument (self, arg, is_variable=False):
        # this formats a single argument - const, type (with scope), name, 
        # annotation, default
        if arg.attributes.cv == 'const':
            args = ['const']
        else:
            args = []
            
        if arg.scope:
            args.append ('%s::%s' % (arg.scope, arg.argumentType))
        else:
            args.append (arg.argumentType)
            
        if arg.argumentName:
            #if is_variable and arg.argumentName:
            args.append (arg.argumentName)
            
        if arg.annotation:
            args.append ('/%s/' % ", ".join (arg.annotation))
            
        if arg.defaultValue:
            args.append ('= %s' % arg.defaultValue)
            
        return " ".join (args)        
        
    def accessChanged (self, obj):
        # detect changes in access and trigger writing a new specifier
        oldAccess = self.access
        if obj.objectType == 'class':
            nextAccess = 'private'
        elif obj.objectType == 'struct':
            nextAccess = 'public'
        elif obj.objectType == 'endclass':
            self.access = nextAccess = None
        else:
            nextAccess = obj.access
        
        self.access = obj.access
            
        return self.access and self.access != oldAccess, nextAccess
        
    def forceChanged (self, obj):
        # detect changes in //force
        if obj.force != self.force:
            self.force = obj.force
            return True
        return False
        
    def format_namespace (self, obj):
        # empty?
        lines = []
        if obj.skip:
            for block in obj.blocks:
                if block.name != '%ConvertToSubClassCode' and block.block and not '//ctscc' in block.block:
                    lines.append ('%s\n' % block.block)
            return lines
            
        # a namespace header
        if obj.ignore:
            lines.append ("//ig namespace %s;\n" % obj.name)
        else:
            self.stateInfo.pushNamespace (obj.name, obj)
            self.docWrite.newNamespace (obj)        
            lines.append ("namespace %s\n{" % obj.name)
    
        for block in obj.blocks:
            if block.name != '%ConvertToSubClassCode' and block.block and not '//ctscc' in block.block:
                lines.insert (0, '%s\n' % block.block)
                
        return lines

    def format_class (self, obj):
        # a class header and %TypeHeaderCode block if not
        # found in sip file already
        self.docWrite.newClass (obj)
        lines = []    
        
        if obj.ignore:
            lines = ['\n//ig class %s;\n' % obj.name]
            if not obj.opaque:
                self.skipClass = 1
            return lines
            
        elif obj.templateParams:
            lines +=  (self.formatTemplate (obj.templateParams))
            hdr = []
            
        elif obj.undefBase:
            lines.append ('//ub class has undefined base class')
            hdr = ['//ub ']
            
        else:
            hdr = ['\n']

        hdr.append ('%sclass %s' % (self.indent (), obj.name))
    
        if obj.opaque or obj.ignore:
            hdr.append (';')
            return ["".join (hdr)]
    
        bases = [base for base in obj.bases if base not in self.modData.nobase]
        if bases:
            hdr.append (' : %s' % ", ".join (bases))
            
        if obj.annotation:
            hdr.append (' /%s/' % ", ".join (obj.annotation))
            
        lines.append ("".join (hdr))
        lines.append ('%s{' % self.indent ())
        
        hasTypeHdr = False
        for block in obj.blocks:
            hasTypeHdr = hasTypeHdr or block.name == '%TypeHeaderCode'
            # things that belong outside the class decl block
            if block.name.startswith ('%Module'):
                lines.insert (0, '%s\n' % block.block)
            elif block.name != '%ConvertToSubClassCode':
                lines.append ('%s\n' % block.block)
        
        if not hasTypeHdr and not self.stateInfo.classStack:
            lines.append ("%TypeHeaderCode")
            lines.append ("#include <%s/%s>" % (self.modData.modname,self.filename.replace ('sip', 'h')))
            lines.append ("%End")
            
        self.stateInfo.pushClass (obj.name, obj)
        self.indentCol += 4
        self.nameIndent += 4
        return lines
    
    def format_endclass (self, obj):
        # close a class
        self.docWrite.endClass (obj)
        self.indentCol -= 4
        self.nameIndent -= 4
        
        lines = ['%s};   // %s\n' % (self.indent (), self.stateInfo.popClass (True))]

        for block in obj.blocks:
            if block.name != '%ConvertToSubClassCode':
                lines.append ('%s\n' % block.block)
        
        return lines
    
    def format_endnamespace (self, obj):
        # empty?
        lines = []
        if obj.skip:
            for block in obj.blocks:
                if block.name != '%ConvertToSubClassCode':
                    lines.append ('%s\n' % block.block)
            return lines
                    
        # close a namespace
        self.docWrite.endNamespace (obj)
        
        lines = ['};   // %s\n' % self.stateInfo.popNamespace (True)] 
        
        for block in obj.blocks:
            if block.name != '%ConvertToSubClassCode':
                lines.append ('%s\n' % block.block)

        return lines
    
    def format_function (self, obj):
        # format a function, including any comment directives
        self.docWrite.addObject (obj)
        lines = []
        isPureVirtual = obj.attributes.functionQualifier == 'pure'
        mcNeeded      = obj.attributes.mcNeeded
        mtNeeded      = obj.attributes.mtNeeded
        ioNeeded      = obj.attributes.ioNeeded        
        indent        = self.indent
        
        comments = []
        if obj.ignore:
            comments = ['ig']
        else:
            comments = []
            
        allArgs = obj.arguments + obj.returns

        if mcNeeded:
            hasMC = False
            for block in obj.blocks:
                if block.name =='%MethodCode':
                    hasMC = True
                    break
            
            if not hasMC:
                for arg in allArgs:
                    if arg.attributes.mcNeeded:
                        lines.append ('//mc symbol: %s %s -- requires %%MethodCode?' % (arg.argumentType, arg.argumentName))       
                comments.append ('mc')
            
        if mtNeeded:
            for arg in allArgs:
                if arg.attributes.mtNeeded:
                    lines.append ('//mt undefined symbol: %s %s -- need mapped type?' % (arg.argumentType, arg.argumentName))
            comments.append ('mt')
            
        if ioNeeded:
            for arg in obj.arguments:
                if arg.attributes.ioNeeded:
                    lines.append ('//io ambiguous symbol: %s %s -- specify In, Out or both' % (arg.argumentType, arg.argumentName))
            comments.append ('io')
        
        f = []
        if comments:
            f.append ('//%s ' % ' '.join (comments))
        
        if obj.attributes.storageClass in ['static', 'extern']:
            f.append ('%s%s ' % (indent (), obj.attributes.storageClass))
        elif obj.attributes.functionQualifier in ['virtual', 'pure']:
            f.append ('%svirtual ' % indent ())
        elif obj.returns [0].argumentType == 'ctor' and obj.attributes.functionQualifier == 'explicit':
            f.append ('%sexplicit ' % indent ())
            #f.append ('%s ' % indent ())
        else:
            f.append (indent ())
            
        if not obj.returns [0].argumentType in ['ctor', 'dtor']:
            f.append (self.formatArgument (obj.returns [0]))
            
        if obj.returns [0].argumentType == 'dtor':
            f.append ('%s~%s (' % (self.gap ("".join (f)), obj.name))
        else:
            f.append ('%s%s (' % (self.gap ("".join (f)), obj.name))
        
        arglist = []
        for arg in obj.arguments:
            arglist.append (self.formatArgument (arg))
            
        f.append ('%s)' % ', '.join (arglist))
        
        if obj.attributes.cv == 'const':
            f.append (' const')
            
        if obj.exceptions:
            f.append (' throw (%s)' % obj.exception)
        
        if isPureVirtual:
            f.append (' = 0')
        
        if obj.annotation:
            f.append (' /%s/' % ", ".join (obj.annotation))
        
        if obj.cppReturns or obj.cppArguments:
            f.append (' [')
            if not obj.returns [0].argumentType == 'ctor':
                f.append ('%s ' % obj.cppReturns [0].argumentType)
        
            arglist = []
            for arg in obj.cppArguments:
                arglist.append (self.formatArgument (arg))
                
            f.append ('(%s)]' % ', '.join (arglist))            

        f.append (';')
        
        lines.append (''.join (f))
                
        if mcNeeded and not hasMC:
            lines.append ('//%MethodCode')
            lines.append ('//%sPy_BEGIN_ALLOW_THREADS' % indent ())
            lines.append ('//%sPy_END_ALLOW_THREADS' % indent ())
            lines.append ('//%End')

        for block in obj.blocks:
            if block.name != '%ConvertToSubClassCode':
                lines.append ('%s\n' % block.block)
        
        commentSpace = comments and (len (comments) > 1 or comments [0] != 'ig')
        if commentSpace:
            lines.insert (0, '\n')
            lines.append ('\n')
        return lines
        
    def format_enum (self, obj):
        # format an enum
        self.docWrite.addObject (obj)
        lines = []
        indent = self.indent
        
        if obj.name == 'anonymous':
            name = ''
        else:
            name = obj.name
            
        lines.append ("\n%senum %s" % (indent (), name))
        if obj.ignore:
            lines [0] = "\n//ig%s;" % lines [0][1:]
            return lines
            
        lines.append ("%s{" % indent ())
        for e in obj.enumerators:
            lines.append ('%s%s,' % (indent () + '    ', e.name))
        lines [-1] = lines [-1].replace (',', '')
        lines.append ('%s};\n' % indent ())        

        for block in obj.blocks:
            if block.name != '%ConvertToSubClassCode':
                lines.append ('%s\n' % block.block)

        return lines
    
    def format_typedef (self, obj):
        # format a typedef
        lines = []
        self.docWrite.addObject (obj)
        if obj.ignore:
            td = ["//ig "]
        elif obj.attributes.mtNeeded:
            lines.append ('//mt undefined symbol: %s -- need mapped type?' % (obj.argumentType))
            td = ['//mt ']
        else:
            td = []
            
        if obj.functionPtr:
            if obj.functionPtr == ['()']:
                obj.functionPtr = []

            td.append ("%stypedef %s %s)(%s);\n" % (self.indent (), obj.argumentType, obj.name, ', '.join (obj.functionPtr)))            
    
        else:
            td.append ("%stypedef %s %s;\n" % (self.indent (), obj.argumentType, obj.name))
            
        lines.append (''.join (td))

        for block in obj.blocks:
            if block.name != '%ConvertToSubClassCode':
                lines.append ('%s\n' % block.block)

        return lines
    
    def format_variable (self, obj):
        # format a variable
        self.docWrite.addObject (obj)
        
        lines = []
        
        if obj.ignore:
            v = ['//ig' + self.indent ()]
        else:
            v = [self.indent ()]
            
        if obj.variable.attributes.storageClass in ['static', 'extern']:
            v.append (obj.variable.attributes.storageClass + ' ')

        if obj.functionPtr:
            v.append (obj.functionPtr)
        else:
            v.append (self.formatArgument (obj.variable, True))

        if obj.annotation:
            v.append (' /%s/' % ", ".join (obj.annotation))
            
        v.append (';')
                   
        lines.append (''.join (v))
        
        for block in obj.blocks:
            if block.name != '%ConvertToSubClassCode':
                lines.append ('%s\n' % block.block)
        
        return lines
        
    def format_sipBlock (self, obj):        
        # a sip block (eg - %MethodCode)
        lines = ['\n']
        if not '//ctscc' in obj.block:
            lines.append (obj.block)
        return lines
    
    def format_sipDirective (self, obj):
        # %If and others
        lines = []
        return lines

    def format_mappedtype (self, obj):
        # A mapped type block
        lines = ['\n']
        if obj.template:
            lines.append ('template <%s>' % obj.template.params)
        lines.append (obj.block)
        return lines
    
    def format_exception (self, obj):
        # an exception - fix!!
        lines = []
        return lines
    
