#     Copyright 2007-8 Jim Bublitz <jbublitz@nwinternet.com>
#     Copyright 2008   Simon Edwards <simon@simonzone.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

import os, os.path, datetime
from plugins.reducetxt import Reduce
import shutil
import glob
import cpplexer
import stateInfo

reducer = Reduce()

class DocFileWriter(object):
    def __init__ (self, symbolData, modData, stateInfo, project=None):
        self.symbolData = symbolData
        self.modData    = modData
        self.stateInfo  = stateInfo
        self.project = project
        
        self.dst     = os.path.join (self.modData.sipDst, 'doc')
        self.htmldst = os.path.join (self.dst, 'html')
        self.moduledst = os.path.join (self.htmldst, self.modData.modname)
        
        if not os.path.exists (self.dst):
            os.mkdir (self.dst)
            os.mkdir (self.htmldst)

        if not os.path.exists (self.moduledst):
            os.mkdir (self.moduledst)
            
        self.mainIndex = open (os.path.join (self.htmldst, 'index.txt'), 'a')
            
        self.currentModule    = self.modData.modname

        self.ctModules   = 0
        self.ctNs        = 0
        self.ctClasses   = 0
        self.ctMethods   = 0
        self.ctEnums     = 0
        self.ctVariables = 0
        self.ctOperators = 0

        self.namespaces   = {}
        self.nsStk        = ['global']
        self.classes      = {}
        self.classStk     = []
        self.currentClass = ""
        self.flagged      = {}

        self.currentNamespace = "global"

        self.namespaces ["global"] = [None] 

    ########################################################################
    # Start event handling methods
    
    def closeModule (self):
        self.mainIndex.close ()
        # create module index, allclasses
 
    def newNamespace (self, obj):
        if not obj.scope:
            self.currentNamespace = obj.name.replace("::",".")
        else:
            self.currentNamespace = '.'.join ([obj.scope, obj.name])
            
        if self.currentNamespace not in self.namespaces:
            self.namespaces [self.currentNamespace] = [obj]
            self.ctNs += 1
        
        self.nsStk.append (self.currentNamespace)
        
    def endNamespace (self, obj):
        self.nsStk.pop ()
        self.currentNamespace = self.nsStk [-1]
        self.currentClass = ''
        
    def newClass (self, obj):
        if not obj.scope:
            classname = obj.name
        else:
            classname = '.'.join ([obj.scope.replace("::","."), obj.name])

        if not obj.ignore:            
            self.classStk.append (classname)
            self.currentClass = classname
            self.classes [classname] = [obj]
            self.ctClasses += 1
        
        self.namespaces [self.currentNamespace].append (obj)
        
    def endClass (self, obj):
        self.classStk.pop ()
        if self.classStk:
            self.currentClass = self.classStk [-1]
        else:
            self.currentClass = ''
            
    def addFlagged (self, obj):
        enum = obj.argumentType [7:-1]
        if '::' in enum:
            enum = enum.split ('::')[1]
        flags = obj.name
        if not self.currentClass in self.flagged:
            self.flagged [self.currentClass] = {}
            
        self.flagged [self.currentClass][enum] = flags
            
    def addObject (self, obj):
        if self.currentClass:
            if obj.objectType == 'typedef' and obj.argumentType.startswith ('QFlags'):
                self.addFlagged (obj)
            else:
                self.classes [self.currentClass].append (obj)
                if obj.objectType == 'function' and obj.attributes.functionQualifier == 'pure':
                    self.classes [self.currentClass][0].abstract = True                
        else:
            self.namespaces [self.currentNamespace].append (obj)
        
    # End event handling methods.
    ########################################################################
    # Format a block of doxygen text.
    def formatDoxygen(self, txt):
        if txt:
            return reducer.do_txt (txt)
        else:
            return ''

    def objSort (self, o1, o2):
        return cmp (o1.name, o2.name)
        
    def objLists (self, objlist):
        enumList     = []
        methodList   = []
        classList    = []
        variableList = []
        
        for obj in objlist:
            if obj.objectType == 'enum':
                enumList.append (obj)
            elif obj.objectType == 'function':
                methodList.append (obj)
            elif obj.objectType == 'class':
                classList.append (obj)
            elif obj.objectType == 'variable':
                variableList.append (obj)
                
        enumList.sort (self.objSort)
        methodList.sort (self.objSort)
        classList.sort (self.objSort)
        variableList.sort (self.objSort)
        
        return enumList, methodList, classList, variableList
        
    def writeMainIndex (self):
        ns = self.namespaces.keys ()
        ns.sort ()
        
        self.mainIndex.write ('m:%s\n' % self.currentModule)
        for n in ns:
            if len (self.namespaces [n]) > 1:
                self.mainIndex.write ('n:%s\n' % n.replace ('::', '.'))
            classes = [(obj.scope, obj.name) for obj in self.namespaces [n] if obj and obj.objectType == 'class']
            classes.sort ()
            for c in classes:
                if c [0]:
                    self.mainIndex.write ('c:%s\n' % '.'.join ([c [0], c [1]]).replace ('::', '.'))
                else:
                    self.mainIndex.write ('c:%s\n' % c [1])
                
        self.mainIndex.close ()
        self.writeModuleIndexPage()
        
    def writeModuleIndexPage (self):
        nsNames = self.namespaces.keys ()
        nsNames.sort ()
        
        if self.modData.maindox:
            maindox = FetchDox(self.modData.maindox)
        else:
            maindox = ''

        page = open (os.path.join (self.moduledst, 'index.html'), 'w')
        page.write (htmlHeader % {'title': ('Module %s' % self.currentModule), 'path': '../'})
        
        page.write("<h1>%s Module</h1>\n" % self.currentModule)
        page.write("<hr>")
        
        page.write(self.formatDoxygen(maindox))
        
        self.writeNSNamespacesIndex(page, nsNames)
            
        # Collect all of the classes, functions etc.
#        enums = []
#        functions = set()
        classes = []
#        variables = []
        for n in nsNames:
            enumList, functionList, classList, variableList = self.objLists(self.namespaces[n][1:])
#            enums.extend(enumList)
#            functions.update(functionList)
            classes.extend(classList)
#            variables.extend(variableList)
            
        self.writeNSClassIndex(page, classes)
        
#        page.write("enums: "+repr(enums))
#        functions = list(functions)

#        page.write("functions: "+ ", ".join( [x.name+" ("+str(id(x))+" "+str(x.scope)+")" for x in functions] ))
#        self.writeNSFunctionIndex(page, functions)
        
#        page.write("variables: "+repr(variables))
        
        page.write(htmlFooter % {'path': '../'})
        page.close()
        
    def copyExtraResources(self):
        if self.project is None:
            return
        if self.project.prj_apidoxcommon is not None:
            self.globCopy(os.path.join(self.project.prj_apidoxcommon,"*"), os.path.join(self.htmldst,"common"))
        if self.project.prj_apidoximage is not None:    
            self.globCopy(os.path.join(self.project.prj_apidoximage,"*.png"), os.path.join(self.htmldst,"images"))
        
    def globCopy(self,src,dest):
        if not os.path.exists(dest):
            os.mkdir(dest)
        for srcfile in glob.glob(src):
            shutil.copy(srcfile,dest)
        
    def writeAllClasses(self):
        nsNames    = []
        classNames = []
        
        self.copyExtraResources()
        
        self.mainIndex.close ()
        print("Reading index files:" + os.path.join (self.htmldst, 'index.txt'))
        self.mainIndex = open (os.path.join (self.htmldst, 'index.txt'), 'r')
        
        for line in self.mainIndex:
            specifier, name = line.split (':')
            name = name.strip ()
            if specifier == 'm':
                module = name
            elif specifier == 'n':
                if name in ['global', 'Sonnet']:
                    nsNames.append ((module, name, '%s - %s' % (name, module)))
                else:
                    nsNames.append ((module, name, name))
            elif specifier == 'c':
                classNames.append ((module, name, name))
        
        self.mainIndex.close ()
        
        def key(x): return x[1]
        nsNames.sort(key=key)
        classNames.sort(key=key)
        
        page = open(os.path.join (self.htmldst, 'allclasses.html'), 'w')
        page.write(htmlHeader % {'title': 'PyKDE Namespace and Class Index', 'path': ''})
        
        # Namespaces
        page.write("<p>\n<h2>All Namespaces</h2>\n")
        
        def extract_name(x):
            nameparts = x.split('.')
            indexstring = nameparts[-1].lower()
            if indexstring.startswith('k'):
                indexstring = indexstring[1:]
            return indexstring
            
        def format(t):
            nameparts = t[1].split('.')
            indexstring = nameparts[-1].lower()
            if indexstring.startswith('k'):
                indexstring = indexstring[1:]
                
            name = nameparts[-1] + '&nbsp;(' + ('.'.join( [t[0]] + nameparts[:-1] )) + ')'
            
            cellcontents = '<a href="' + t[0] + '/' + t[1] + '.html">' + name + '</a>'
            return (indexstring,cellcontents)
        def t_cmp(a,b):
            return kmp(extract_name(a[1]), extract_name(b[1]))
        page.write(FormatTable(nsNames, format, t_cmp, columns=2))

        # Classes
        page.write("<p>\n<h2>All Classes</h2>\n")
        page.write(FormatTable(classNames, format, t_cmp, columns=2))

        page.write(htmlFooter % {'path': ''})
        page.close()
       
        self.writeMainPage()
        
    def writeMainPage(self):
        page = open(os.path.join (self.htmldst, 'index.html'), 'w')
        page.write(htmlHeader % {'title': 'KDE 4.1 PyKDE API Reference', 'path': ''})
        
        page.write("""<p>
<h2>KDE 4.1 PyKDE API Reference</h2>
</p>
<p>
This is the reference to the KDE API as it appears to Python programs using PyKDE. This is just
reference material, additional information about developing KDE applications using Python and the KDE API
in general can be found on <a href="http://techbase.kde.org/">Techbase</a> and in the 
<a href="http://techbase.kde.org/Development/Languages/Python">Python section</a>. The reference for
<a href="http://www.riverbankcomputing.co.uk/static/Docs/PyQt4/html/classes.html">PyQt 4 is here</a>.
</p>

<p>
This documentation is automatically generated from the original C++ headers and embedded documentation.
The classes, methods, functions, namespaces etc are described as they appear to Python code and
also include addition type information about expected parameters and return types. Note that 
code fragments in the documentation have not been translated from C++ to Python.
</p>
""")
        
        page.write(htmlFooter % {'path': ''})
        
        page.close()
       
    def writeNamespaces (self):
        for ns in self.namespaces:
            self.writeNamespacePage(ns)

    def writeNamespacePage(self,ns):
        nsObj   = self.namespaces [ns][0]
        if len (self.namespaces [ns]) < 2:
            return
            
        enumList, methodList, classList, variableList = self.objLists (self.namespaces [ns][1:])

        ns = ns.replace("::",".")
        # create namespace page and add header
        nspage = open(os.path.join (self.moduledst, "%s.html" % ns), 'w')
        nspage.write(htmlHeader % {'title': ns, 'path': '../'})
        
        importcode = "from PyKDE4.%s.%s import *" % (self.currentModule,ns)
        if ns=='global':
            importcode = "from PyKDE4." + self.currentModule + " import *"
        nspage.write(namespaceHeader % {'namespacename': ns,
            'modulename': self.currentModule,
            'import': importcode,
            'description': self.formatDoxygen(nsObj.doc) if nsObj is not None else ''})
            
        # Class index (to other pages)
        self.writeNSClassIndex(nspage, classList)
        
        # Intra-page index
        nspage.write("""<table border="0" cellpadding="0" cellspacing="0">""")
        
        # enums
        nspage.write(self.writeEnums(enumList))
        
        # Functions
        count = [0]
        def FunctionFilter(obj):
            count[0] += 1
            return True
        s = self.writeMethodIndex(methodList, FunctionFilter, "Functions", False)
        if count[0]!=0:
            nspage.write(s)

        # Variables
        nspage.write(self.writeVariables(variableList))
        
        nspage.write("</table>\n")

        # enum detail
        self.writeEnumDetail (nspage, enumList)
        
        # methods
        self.writeMethodDetails(nspage, methodList, "Function Documentation", True)
        
        # variables
        self.writeVariableDetails(nspage, variableList)
        
        # footer
        nspage.write (htmlFooter % {'path': '../'})
        nspage.close ()
        
    def writeNSClassIndex(self, page, rawClassList):
        if not rawClassList:
            return
            
        page.write ("<h2>Class Index</h2>\n")

        def format(obj):
            if not obj.ignore:
                classname = indexstring = obj.name
                if indexstring[0].upper()=='K':
                    indexstring = indexstring[1:]
                if not obj.scope:
                    cellcontents = '<a class="el" href="%s.html">%s</a>&nbsp;&nbsp;&nbsp;' \
                                    % (classname,classname)
                else:
                    scope = obj.scope.replace("::",".")
                    cellcontents = '<a class="el" href="%s.%s.html">%s</a>&nbsp;(<a class="el" href="%s.html">%s</a>)&nbsp;&nbsp;&nbsp;' \
                                    % (scope,classname,classname,scope,scope)
                    
                return (indexstring,cellcontents)
            else:
                return None

        page.write(FormatTable(rawClassList, format, kmp_obj))

    def writeNSNamespacesIndex(self, page, namespaces):
        if not namespaces:
            return
        
        page.write("<h2>Namespaces</h2>\n")
        
        def format(obj):
            indexstring = obj
            if indexstring[0].upper()=='K':
                indexstring = indexstring[1:]
            name = obj
            if name=='global':
                name = '<i>' + name + '</i>'
            cellcontents = '<a class="el" href="%s.html">%s</a>&nbsp;&nbsp;&nbsp;' % (obj,name)
            return (indexstring,cellcontents)
        page.write(FormatTable(namespaces, format, kmp))
    
    def writeNSFunctionIndex(self, page, functions):
        if not functions:
            return
        page.write("<h2>Functions</h2>\n")
        
        def format(obj):
            if obj.scope is not None:
                name = pyName(obj) + "(" + obj.scope + ")"
            else:
                name = pyName(obj)
            return (pyName(obj),pyName(obj))
        def functionsort(a,b):
            return cmp(pyName(a),pyName(b))
        
        page.write(FormatTable(functions, format, functionsort))
    
    def writeClasses (self):
        for cls in self.classes:
            self.writeClassPage(cls)
            
    def writeClassPage(self,cls):
        # sort objects            
        clsObj = self.classes [cls][0]
        enumList, methodList, classList, variableList = self.objLists (self.classes [cls][1:])
        if cls in self.flagged:
            flags = self.flagged [cls]
        else:
            flags = None
        
        # create class page and add header
        if clsObj.scope:
            classname = '.'.join ([clsObj.scope.replace("::","."), pyName(clsObj)])
        else:
            classname = pyName(clsObj)
        page = open (os.path.join (self.moduledst, "%s.html" % classname), 'w')
        
        page.write (htmlHeader % {'title': classname, 'path': '../'})
        
        if clsObj.abstract:
            abstract = """<dl class="abstract" compact><dt><b>Abstract class:</b></dt>
<dd>This class can be used as a base class for new classes, but can not be instantiated directly.</dd></dl>"""
        else:
            abstract = ''
        # &#x2192;  arrow
        if len(clsObj.bases)!=0:
            bases = []
            for base in clsObj.bases:
                bases.append(' &#x2192; '.join([self.linkType(x) for x in [base]+self.symbolData.hierarchy.getSuperclasses(base)]))
            
            baseclasses = 'Inherits: '+ (','.join(bases)) + '<br />'
        else:
            baseclasses = ''
        
        subclassset = self.symbolData.hierarchy.getSubclasses(pyName(clsObj))
        if len(subclassset)!=0:
            subclasslist = list(subclassset)
            subclasslist.sort()
            subclasses = 'Subclasses: ' + (', '.join ( [self.linkType(x) for x in subclasslist] )) + '<br />'
        else:
            subclasses = ''
        
        page.write (classHeader %
            {'classname': pyName(clsObj),
            'abstract': abstract,
            'modulename': self.currentModule,
            'namespace': 'Namespace: '+clsObj.scope + '<br />' if clsObj.scope!='' else '',
            'baseclasses': baseclasses,
            'subclasses': subclasses,
            'description': self.formatDoxygen(clsObj.doc)} )
            
        page.write("""<table border="0" cellpadding="0" cellspacing="0">""")
        
        # enums
        page.write(self.writeEnums (enumList, flags))
        
        page.write(self.writeVariables(variableList))
        
        count = [0]
        def SignalFilter(obj):
            if obj.access=='signals':
                count[0] += 1
                return True
            else:
                return False
        s = self.writeMethodIndex(methodList, SignalFilter, "Signals", False)
        if count[0]!=0:
            page.write(s)
        
        count = [0]
        def MethodFilter(obj):
            if 'static' not in obj.attributes.storageClass:
                count[0] += 1
                return True
            else:
                return False
        s = self.writeMethodIndex(methodList, MethodFilter, "Methods")
        if count[0]!=0:
            page.write(s)
        
        count = [0]
        def StaticFilter(obj):
            if 'static' in obj.attributes.storageClass:
                count[0] += 1
                return True
            else:
                return False
        s = self.writeMethodIndex(methodList, StaticFilter, "Static Methods", False)
        if count[0]!=0:
            page.write(s)

        page.write("</table>\n")
        
        # methods
        self.writeMethodDetails(page, methodList)
        
        # variables
        self.writeVariableDetails(page, variableList)
        
        # enum detail
        self.writeEnumDetail (page, enumList, flags)
    
        # footer
        page.write (htmlFooter % {'path': '../'})
        page.close ()

    # formatType()
    # rObj - string, type name, may contain C++ '&' and '*' characters.
    def formatType(self, rObj):
        ret = rObj.argumentType
        r = ''
        if not ret or ret == 'void':
            return ''
            
        if ret=="char*":
            ret = "QString"
            
        for i in range (len (ret)):
            if not ret [i] in ['*', '&']:
                r += ret [i]
            else:
                break
        
        if rObj.scope:
            r = '.'.join ([rObj.scope.replace("::","."), r])
        r = self.linkType(r)
        return r
        
    CppPythonTypeMapping = {
        'bool': 'bool',
        'int': 'int',
        'float': 'float',
        'long': 'long',
        'qint32': 'int',
        'quint16': 'int',
        'qint64': 'long',
        'quint32': 'long',
        'uint': 'long',
        'qlonglong': 'long',
        'qulonglong': 'long',
        'double': 'float',
        'unsigned int': 'long',
        'unsigned long': 'long',
        'qreal': 'float',
        'unsigned short': 'long'
    }
        
    def linkType(self, typeName):
        # Handle QList<...>
        if typeName.startswith("QList<"):
            return '[' + self.linkType(typeName[6:-1]) + ']'
        if typeName.startswith("Qt.") or typeName.startswith("Qt::"):
            return '<a href="http://www.riverbankcomputing.co.uk/static/Docs/PyQt4/html/qt.html">' + typeName + '</a>'
        
        pyType = DocFileWriter.CppPythonTypeMapping.get(typeName)
        if pyType is not None:
            return pyType
    
        match = self.symbolData.index.getMatchingObjects(typeName,['class'])
        filepath = None
        
        fileName = typeName.replace('::','.')
        for m in match:
            if m.scope is not None and m.scope!="" and '.' not in fileName:
                fileName = m.scope + '.' + fileName
                print("typeName: "+ typeName +", m.scope: " + m.scope)
                
            if self.modData.modname==m.module and m.filepath is None:
                return '<a href="../' + m.module + '/' + fileName + '.html">' + fileName + '</a>'
            if m.filepath is None:
                return '<a href="../' + m.module + '/' + fileName + '.html">' + fileName + '</a>'
            else:
                filepath = m.filepath
            
        if filepath is not None:
            parts = filepath.split('/')
            if parts[-3]=='PyQt4':
                return '<a href="http://www.riverbankcomputing.co.uk/static/Docs/PyQt4/html/' + typeName.lower() + '.html">' + typeName + '</a>'
                
        if typeName != "ctor":
            print(typeName+" -> ?")
        return typeName
   
    def writeMethodDetails(self, page, methodList, title="Method Documentation", functions=False):
        if not methodList:
            return
            
        page.write('<hr><h2>' + title + '</h2>')
        for obj in methodList:
            self.writeMethodDetail(page, obj, function=functions)
        
    def writeMethodDetail(self, page, obj, function=False):
        if obj.ignore or obj.access == 'private':
            color = "#999999"
        else:
            color = "#000000"
            
        retList = []
        if obj.returns:
            for r in obj.returns:
                retList.append (self.formatType(r))
            
        for arg in obj.arguments:
            if arg.annotation and 'Out' in arg.annotation:
                if arg.argumentName:
                    retList.append ('%s (%s)' % (self.formatType(arg), arg.argumentName))
                else:
                    retList.append (self.formatType(arg))
        
        if retList:
            ret = ', '.join (retList)
        else:
            ret = ''
        if ret=='dtor':
            return

#            param = ''
#            if obj.ignore or obj.access == 'private':
#                param = '<i style="color : #aa0000">Not implemented</i>'
        i = 0
        for arg in obj.arguments:
            if not arg.argumentName:
                arg.argumentName = 'a%i' % i
                i += 1                   
            
        args = """<a class="anchor" name="obj""" + str(id(obj)) + """"></a>
<div class="memitem">
<div class="memproto">
<table class="memname">"""
        if ret=="ctor":
            memname = "__init__"
            ret = ""
        else:
            memname = ret + " " + pyName(obj)
        
        if not obj.arguments:
            selfarg = '<em>self</em>&nbsp;' if 'static' not in obj.attributes.storageClass else ''
            args += """<tr>
<td class="memname">%s</td>
<td>(</td>
<td class="paramtype">&nbsp;</td>
<td class="paramname">%s)</td>
<td width="100%%"> </td>
</tr>
""" % (memname,selfarg)
        
        else:
            bracket = "("
            if not function and 'static' not in obj.attributes.storageClass:
                args += """<tr>
<td class="memname">%s</td>
<td>%s</td>
<td class="paramtype">&nbsp;<em>self</em>, </td>
<td class="paramname"></td>
</tr>""" % (memname,bracket)
                memname = ""
                bracket = ""
            
            for a in obj.arguments:
                if not a.annotation or ('In' in a.annotation or not 'Out' in a.annotation):
                    paramtype = self.formatType(a)
                    paramname = a.argumentName
                    if a.defaultValue is not None:
                        paramname += "=" + a.defaultValue.replace ('::', '.')
                        
                    comma = ", " if a is not obj.arguments[-1] else ""
                
                    args += """<tr>
<td class="memname">%s</td>
<td>%s</td>
<td class="paramtype">%s&nbsp;</td>
<td class="paramname"><em>%s</em>%s</td>
</tr>
""" % (memname,bracket,paramtype,paramname,comma)
                    memname = ""
                    bracket = ""
            args += """<tr>
<td></td>
<td>)</td>
<td></td>
<td></td>
<td width="100%"> </td>
</tr>"""
        args += """</table>
</div>
"""
        page.write(args)
    
        page.write('<div class="memdoc">')
        if 'pure' in obj.attributes.functionQualifier:
            page.write("""<dl compact><dt><b>Abstract method:</b></dt><dd>""")
            page.write("This method is abstract and can be overridden but not called directly.")
            page.write("""</dd></dl>""")
        page.write(self.formatDoxygen(obj.doc))
        
        if obj.access == 'signals':
            args = []
            for arg in obj.arguments:
                if arg.attributes.cv == 'const':
                    args.append ('const %s' % arg.argumentType)
                else:
                    args.append (arg.argumentType)
            page.write("""<dl compact><dt><b>Signal syntax:</b></dt><dd>""")
            page.write("""<code>QObject.connect(source, SIGNAL("%s(%s)"), target_slot)</code>""" % (pyName(obj), ', '.join (args)))
            page.write("""</dd></dl>""")
        
        page.write('</div></div>')
        
    def writeMethodIndex(self, methodList, methodFilter, title, useSelf=True):
        if not methodList:
            return ""
        
        result = []
        result.append("""<tr><td colspan="2"><br><h2>""")
        result.append(title)
        result.append("""</h2></td></tr>\n""")
        
        for obj in methodList:
            if not methodFilter(obj):
                continue
            
            if obj.ignore or obj.access == 'private':
                color = "#999999"
            else:
                color = "#000000"
                
            retList = []
            if obj.returns:
                for r in obj.returns:
                    retList.append (self.formatType(r))
                
            for arg in obj.arguments:
                if arg.annotation and 'Out' in arg.annotation:
                    if arg.argumentName:
                        retList.append ('%s (%s)' % (self.formatType(arg), arg.argumentName))
                    else:
                        retList.append (self.formatType(arg))
            
            if retList:
                ret = ', '.join (retList)
            else:
                ret = ''
            if ret=='dtor':
                continue
                
#            param = ''
#            if 'pure' in obj.attributes.functionQualifier:
#                param += '<b style="color : #00aa00">pure virtual</b>'
#            if obj.ignore or obj.access == 'private':
#                param = '<i style="color : #aa0000">Not implemented</i>'
#            
#            if param != '':
#                print(obj.name + "~> "+param)
                
            if obj.access == 'signals':
                args = []
                for arg in obj.arguments:
                    if arg.attributes.cv == 'const':
                        args.append ('const %s' % arg.argumentType)
                    else:
                        args.append (arg.argumentType)

            i = 0
            for arg in obj.arguments:
                if not arg.argumentName:
                    arg.argumentName = 'a%i' % i
                    i += 1                   
                
            if ret=="ctor":
                memname = "__init__"
                ret = ""
            else:
                memname = pyName(obj)
            memname = '<a class="el" href="#obj' + str(id(obj)) + '">' + memname + '</a>'
                
            result.append('<tr><td class="memItemLeft" nowrap align="right" valign="top">' + ret +'&nbsp;</td><td class="memItemRight" valign="bottom">' + memname + ' (')
            
            if useSelf:
                result.append('self')
                if len(obj.arguments)!=0:
                    result.append(", ")
                    
            for a in obj.arguments:
                if not a.annotation or ('In' in a.annotation or not 'Out' in a.annotation):
                    paramtype = self.formatType(a)
                    paramname = a.argumentName
                    if a.defaultValue is not None:
                        paramname += "=" + a.defaultValue.replace ('::', '.')
                        
                    comma = ", " if a is not obj.arguments[-1] else ""
                
                    result.append(paramtype + " " + paramname + comma)
            result.append(')</td></tr>\n')
        return "".join(result)
        
    def writeVariables(self, variableList):
        if not variableList:
            return ""
        result = []
        result.append("""<tr><td colspan="2"><br><h2>""")
        result.append("Attributes")
        result.append("""</h2></td></tr>\n""")
        
        for obj in variableList:
            result.append('<tr><td class="memItemLeft" nowrap align="right" valign="top">' + self.formatType(obj.variable) + '&nbsp;</td><td class="memItemRight" valign="bottom"><a class="el" href="#var' + str(id(obj)) + '">' + obj.name +'</a></td></tr>')
        return "".join(result)
    
            
    def writeEnums (self, enumList, flags = None):
        if not enumList:
            return ""
            
        result = []
        result.append("""<tr><td colspan="2"><br><h2>Enumerations</h2></td></tr>\n""")

        for obj in enumList:
            typeSafe = None
            if not obj.name:
                objName = "&lt;anonymous&gt;"
            else:
                if flags and obj.name in flags:
                    typeSafe = flags [obj.name]
                objName = obj.name
           
            result.append('<tr><td class="memItemLeft" nowrap align="right" valign="top"><a class="el" href="#obj' + str(id(obj)) + '">' + objName +'</a>&nbsp;</td><td class="memItemRight" valign="bottom">{&nbsp;')
               
            result.append(", ".join( [x.name for x in obj.enumerators] ))
                
            if typeSafe:
                result.append('&nbsp;}<br><i>Typesafe wrapper:</i> %s</td></tr>\n' % (typeSafe))
            else:
                result.append("&nbsp;}</td></tr>\n")

        return "".join(result)
        
    def writeVariableDetails (self, page, variableList):
        if not variableList:
            return
        
        page.write('<hr><h2>Attribute Documentation</h2>')

        for obj in variableList:
            page.write("""<a class="anchor" name="var""" + str(id(obj)) + """"></a>
<div class="memitem">
<div class="memproto">
<table class="memname">
<tr><td class="memname">""")
            page.write(self.formatType(obj.variable))
            page.write(" ")
            page.write(pyName(obj))
            page.write("""</td>
</tr>
</table>
</div>
<div class="memdoc">""")
            
            if obj.doc:
                doc = obj.doc
            else:
                doc = ''
            page.write(self.formatDoxygen(doc))
            
            page.write("""</div></div><p>""")            
            
    def writeEnumDetail (self, page, enumList, flags = None):
        if not enumList:
            return
            
        page.write('<hr><h2>Enumeration Documentation</h2>')
        
        for obj in enumList:
            page.write("""<a class="anchor" name="obj""" + str(id(obj)) + """"></a>
<div class="memitem">
<div class="memproto">
<table class="memname">
<tr><td class="memname">""")
            typeSafe = None
            if not obj.name:
                name = 'anonymous'
            else:
                if flags and obj.name in flags:
                    typeSafe = flags [obj.name]
                name = obj.name
            page.write(name)
            page.write("""</td>
</tr>
</table>
</div>
<div class="memdoc">""")
            
            if obj.doc:
                doc = obj.doc
            else:
                doc = ''
            page.write(self.formatDoxygen(doc))
            
            if typeSafe:
                page.write("""<dl compact><dt><b>Note:</b></dt><dd>""")
                page.write("It is necessary to wrap members of this enumeration in a <code>" + typeSafe + "</code> instance when passing them to a method as group of flags. ")
                if len(obj.enumerators)>=2:
                    page.write("For example: <code>" + typeSafe + "( " + obj.enumerators[0].name + " | " + obj.enumerators[1].name + ")</code>")
                page.write("""</dd></dl>""")
                
            page.write("""<dl compact><dt><b>Enumerator: </b></dt><dd>
<table border="0" cellspacing="2" cellpadding="0">""")

            for e in obj.enumerators:
                docLine = self.formatDoxygen(e.doc).strip().replace('/', ' ')
                if e.value:
                    val = '=&nbsp;' + str(e.value)
                else:
                    val = ''
                
                page.write('<tr><td valign="top"><em>')
                page.write(e.name)
                page.write('</em>&nbsp;')
                page.write(val)
                page.write(docLine)
                page.write('</td><td>')
            page.write("""</table>
</dl>
</div></div><p>""")

# name
htmlHeader = """<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html xmlns="http://www.w3.org/1999/xhtml" lang="en" xml:lang="en">

<head>
  <title>%(title)s</title>
  <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
  <meta http-equiv="Content-Style-Type" content="text/css" />
  <link rel="stylesheet" type="text/css" href="%(path)scommon/doxygen.css" />
  <link rel="stylesheet" media="screen" type="text/css" title="KDE Colors" href="%(path)scommon/kde.css" />
</head>
<body>
<div id="container">
<div id="header">
  <div id="header_top">
    <div>
      <div>
        <img alt ="" src="%(path)scommon/top-kde.jpg"/>
        KDE 4.1 PyKDE API Reference
      </div>
    </div>
  </div>
  <div id="header_bottom">
    <div id="location">
      <ul>
        <li>KDE's Python API</li>
      </ul>
    </div>

    <div id="menu">
      <ul>
        <li><a href="%(path)sindex.html">Overview</a></li>
<li><a href="http://techbase.kde.org/Development/Languages/Python">PyKDE Home</a></li>
<li><a href="http://kde.org/family/">Sitemap</a></li>
<li><a href="http://kde.org/contact/">Contact Us</a></li>
</ul>
    </div>
  </div>
</div>

<div id="body_wrapper">
<div id="body">
<div id="right">
<div class="content">
<div id="main">
<div class="clearer">&nbsp;</div>
"""

# nothing
htmlFooter = """
</div>
</div>
</div>

<div id="left">

<div class="menu_box">
<div class="nav_list">
<ul>
<li><a href="%(path)sallclasses.html">Full Index</a></li>
</ul>
</div>

<a name="cp-menu" /><div class="menutitle"><div>
  <h2 id="cp-menu-project">Modules</h2>
</div></div>
<div class="nav_list">
<ul>
<li><a href="%(path)sakonadi/index.html">akonadi</a></li>
<li><a href="%(path)sdnssd/index.html">dnssd</a></li>
<li><a href="%(path)skdecore/index.html">kdecore</a></li>
<li><a href="%(path)skdeui/index.html">kdeui</a></li>
<li><a href="%(path)skhtml/index.html">khtml</a></li>
<li><a href="%(path)skio/index.html">kio</a></li>
<li><a href="%(path)sknewstuff/index.html">knewstuff</a></li>
<li><a href="%(path)skparts/index.html">kparts</a></li>
<li><a href="%(path)skutils/index.html">kutils</a></li>
<li><a href="%(path)snepomuk/index.html">nepomuk</a></li>
<li><a href="%(path)sphonon/index.html">phonon</a></li>
<li><a href="%(path)ssolid/index.html">solid</a></li>
<li><a href="%(path)ssoprano/index.html">soprano</a></li>
</ul></div></div>

</div>

</div>
  <div class="clearer"/>
</div>

<div id="end_body"></div>
</div>
<div id="footer"><div id="footer_text">
This documentation is maintained by <a href="&#109;&#97;&#105;&#108;&#116;&#111;&#58;simon&#64;simonzone&#46;com">Simon Edwards</a>.<br />
        KDE<sup>&#174;</sup> and <a href="../images/kde_gear_black.png">the K Desktop Environment<sup>&#174;</sup> logo</a> are registered trademarks of <a href="http://ev.kde.org/" title="Homepage of the KDE non-profit Organization">KDE e.V.</a> |
        <a href="http://www.kde.org/contact/impressum.php">Legal</a>
    </div></div>
</body>
</html>
"""

#nsName, module, nsName, text
namespaceHeader = """
<h1>%(namespacename)s Namespace Reference</h1>
<code>%(import)s</code>
<p>
<h2>Detailed Description</h2>
%(description)s
"""

classListEntry = '<a href="%s.html">%s</a>'
igclassListEntry = '<i style="color : #999999">%s</i>'

# classname, abstract, modulename, namespace, classname, baseclasses, description
classHeader = """
<h1>%(classname)s Class Reference</h1>
<code>from PyKDE4.%(modulename)s import *</code>
<p>
%(baseclasses)s
%(subclasses)s
%(namespace)s
<h2>Detailed Description</h2>
%(abstract)s
%(description)s
"""

varEntry   = '<tr><td width="60%%" valign="top"><b>%s</b> <i>%s</i></td><td width="40%%" align="left">%s</td></tr>\n'

# from, to
acColHdr = """    <td valign="top">
      <table border="0">
        <tr>
          <td valign="top">
            <table border="0">
              <tr bgcolor="#fff0ff"><th>%s - %s</th></tr>
"""

# filename, filename
clsColEntry = '              <tr><td><a href="%s.html">%s</a></td></tr>\n'

# module, filename, filename
acColEntry = '              <tr><td><a href="%s/%s.html">%s</a></td></tr>\n'

acColFooter = """        </table>
          </td>
        </tr>
      </table>
    </td>
"""

############################################################################
def kmp(a,b):
    if a.startswith('K'):
        a = a[1:]
    if b.startswith('K'):
        b = b[1:]
    return cmp(a.lower(),b.lower())
    
def kmp_obj(a,b):
    return kmp(a.name,b.name)
    
# (indexstring,cellcontents) = cellFunc(obj)
def FormatTable(rawObjectList, cellFunc, cmpFunc=cmp, columns=3):
    if not rawObjectList:
        return

    # Sort the class list with special treatment for the letter K.
    objectList = rawObjectList[:]
    objectList.sort(cmpFunc)
    
    result = []
    
    cells = []
    letter = None
    for obj in objectList:
        tuple = cellFunc(obj)
        if tuple is None:
            continue
        indexstring, cellcontents = tuple

        newletter = indexstring[0].upper()
        if newletter!=letter:
            letter = newletter
            cells.append('<a name="letter_'+letter+'">&nbsp;&nbsp;'+letter+'&nbsp;&nbsp;</a>')
        cells.append(cellcontents)

    # Write it out.
    result.append("""<table width="95%" align="center" border="0" cellpadding="0" cellspacing="0">
<tbody>""")
    
    def cell(i): return cells[i] if i<len(cells) else ''
    
    section = int((len(cells)+columns-1)/columns)
    for i in range(section):
        result.append("<tr>")
        for j in range(columns):
            result.append("<td>" + cell(j*section + i) + "</td>")
        result.append("</tr>\n")
    result.append("</table>\n")
    
    return "".join(result)

############################################################################
class MainPageDoxStateInfo(stateInfo.StateInfo):
    def __init__(self):
        stateInfo.StateInfo.__init__(self)
        self.all_text = []
    
    def setDoc(self,text):
        self.all_text.append(text)
        stateInfo.StateInfo.setDoc(self,text)
    
def FetchDox(filename):
    si = MainPageDoxStateInfo()
    cpplexer.setStateInfoTarget(si)
    
    f = open(filename)
    text = f.read()
    f.close()
    
    cpplexer.cppLexer.input("class foo { "+text+"};")

    # Tokenize
    while True:
        tok = cpplexer.cppLexer.token()
        if not tok:
            break
    return si.all_text[0]
    
def pyName(obj):
    return obj.name if obj.pyName is None else obj.pyName
