#     Copyright 2007-8 Jim Bublitz <jbublitz@nwinternet.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

import sys, os
    
ifTemplate     = "%sif (dynamic_cast<%s*>(sipCpp))\n"
elseTemplate   = "%selse if (dynamic_cast<%s*>(sipCpp))\n"
assignTemplate = "%ssipClass = sipClass_%s;\n"
indent         = "                                             "

class SubClasses (dict):
    """
    Identify the classes which go in %ConvertToSubClassCode blocks
    and set up structures needed to write those blocks into the sip
    files.
    """
    def __init__ (self, symbolData, modInfo):
        self.modInfo     = modInfo
        self.cast        = modInfo.cast
        self.nocast      = modInfo.nocast
        self.modname     = modInfo.modname
        self.hierarchy   = symbolData.hierarchy
        self.symbolData  = symbolData

        # currentModClasses/currentCls are lists of the classes
        # defined in the current module
        self.currentCls  = self.hierarchy.currentModClasses

        self.modClasses  = {}
        for cls in self.currentCls:
            # is this class excluded from %ConvertToSubClassCode?
            if cls in self.nocast:
                continue
                
            # get the base class for cls
            base = self.hierarchy [cls]

            # find the deepest base either in imported modules
            # or in this module (for nesting in CTSCC blocks)
            while base and base in self.hierarchy and self.hierarchy [base]:
                if base in self.currentCls:
                    break
                base = self.hierarchy [base]

            if base and base in self.cast or base in self.currentCls:
                self.modClasses [cls] = base
 
        # pass 1
        # self.casts is a dictionary from base classes to
        # derived classes - eg QObject: [list of QObject subclasses in this module]
        self.casts = {}
        for cls in self.cast:
            self.casts [cls] = []
            
        # first remove all of the classes in modClasses that
        # are declared in the 'cast' stmt in the prj file
        for key in self.modClasses.keys ():
            base = self.modClasses [key]
            if  base in self.casts:
                self.casts [base].append (key)
                del self.modClasses [key]
                
                
        # passes 2 ... N
        # remove all of the classes in modClasses that go into
        # CTSCC blocks and place those in the lists in self.casts
        # keep going until no more classes can be removed
        flag = True
        while flag:
            flag = False
            for key in self.casts.keys ():          
                for key1 in self.modClasses.keys ():
                    if self.modClasses [key1] in self.casts [key]:
                        flag = True
                        base = self.modClasses [key1]
                        if base not in self.casts:
                            self.casts [base] = [key1]
                        else:
                            self.casts [base].append (key1)
                        del self.modClasses [key1]
                        

        # now self.casts holds a list of classes that go into CTSCC
        # blocks for each base class
        
        # next, we want to create a dict which lists the file the CTSCC
        # block goes in, and the base class that the CTSCC block covers
        
            self.winnow ()
            self.writeCTSCC ()
            
    def winnow (self):
        self.castFiles = {}
        for base in self.casts:
            flag = False
            clsList = self.casts [base]
            if not clsList:
                continue
                
            clsList.sort ()
            newClsList = []
            for cls in clsList:
                matches = self.symbolData.index.getMatchingObjects (cls, ["class"])
                for match in matches:
                    if match.module == self.modname:
                        file = match.filename
                        if file [:-4] in self.symbolData.fileData:
                            for obj in self.symbolData.fileData [file [:-4]]:
                                if obj.objectType == 'class' and obj.name == cls and not obj.ignore:
                                    newClsList.append ((cls, file [:-4]))
                                    if not flag and not file in self.castFiles and base in self.cast:
                                        self.castFiles [file] = (base, cls)
                                        flag = True
                                        
            self.casts [base] = newClsList
                                    
        
    def writeCTSCC (self):
        dst = self.modInfo.sipDst
        for file in self.castFiles:
            modHdrFlag  = False
            base        = self.castFiles [file][0]
            targetClass = self.castFiles [file][1]
            
            filename = os.path.join (dst, 'sip', self.modInfo.modname, file)
            self.sipInFile  = open (filename, 'r')
            self.sipOutFile = open ('%s.tmp' % filename, 'w')
            
            for line in self.sipInFile:
                if  not modHdrFlag\
                    and not line.strip ().startswith ("class")\
                    and not line.strip ().startswith ('namespace'):
                    self.sipOutFile.write (line)                        
                    continue
    
                if not modHdrFlag:
                    self.writeModHdr (self.casts [base])
                    modHdrFlag = True

                self.sipOutFile.write (line)
                if not line.strip ().startswith ('class %s :' % targetClass):
                    continue
                
                lastClsLine = self.findEndClass ()
                if not lastClsLine:
                    print 'Bracket mismatch in file %s' % file
                    sys.exit (-1)
                    
               
                self.sipOutFile.write ('\npublic:\n\n//  Subclasses of %s\n\n%%ConvertToSubClassCode\n\n    sipClass = NULL;\n\n' % base)
                self.writeCTSCCblock (base, self.refine (base))
                self.finish (lastClsLine, filename)
                break
                
    def refine (self, base):
        clsList = [c [0] for c in self.casts [base]]
        for i in range (len (clsList)):
            matches = self.symbolData.index.getMatchingObjects (clsList [i], ["class"])
            for match in matches:
                if match.module == self.modname:
                    if match.scope:
                        clsList [i] = "::".join ([match.scope, clsList [i]])
                        
        return clsList
        
    def buildIncludeList (self, clsList, newList):
        for cls in clsList:
            newList.append (cls)
            if cls [0] in self.casts:
                self.buildIncludeList (self.casts [cls [0]], newList)
        return newList       
            
    def writeModHdr (self, clsList):
        self.sipOutFile.write ('\n%ModuleHeaderCode\n//ctscc\n')
        clsDict = {}
        newList = []
        self.buildIncludeList (clsList, newList)
        
        # eliminate duplicate h files by punching everything into a dict 
        for cls in newList:
            clsDict [cls [1]] = None
        
        clsList = clsDict.keys ()
        clsList.sort ()
        
        for cls in clsList:
            self.sipOutFile.write ('#include <%s.h>\n' % cls)
        self.sipOutFile.write ('%End\n\n')
        
    def findEndClass (self):
        brackets = 0
        
        for line in self.sipInFile:
            if '{' in line:
                brackets += 1
            if '}' in line:
                brackets -= 1
            if not brackets and '};' in line:
                return line
            self.sipOutFile.write (line)
            
        return None
            
    def writeCTSCCblock (self, base, clsList, offset = 4):
        elseFlag = False
            
        for cls in clsList:
            if '::' in cls:
                class_ = cls.split ('::')[-1]
            else:
                class_ = cls
                
            if not elseFlag:
                self.sipOutFile.write (ifTemplate % (indent [:offset], cls))
                elseFlag = True
            else:
                self.sipOutFile.write (elseTemplate % (indent [:offset], cls))
            offset += 4                   
            
            if class_ in self.casts:
                self.sipOutFile.write ('%s{\n' % indent [:offset])
                offset += 4
              
            self.sipOutFile.write (assignTemplate % (indent [:offset], cls.replace ('::', '_')))
            
            if class_ in self.casts:
                self.writeCTSCCblock (base, self.refine (class_), offset)
                offset -= 4
                self.sipOutFile.write ('%s}\n' % indent [:offset])
            
            offset -= 4

    def finish (self, lastClsLine, filename):
        self.sipOutFile.write ('%End\n\n')
        self.sipOutFile.write (lastClsLine)
        for line in self.sipInFile:
            self.sipOutFile.write (line)
            
        self.sipInFile.close ()
        self.sipOutFile.close ()
        os.unlink (filename)
        os.rename ('%s.tmp' % filename, filename)
        
