#     Copyright 2007-8 Jim Bublitz <jbublitz@nwinternet.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

import sys, os, os.path
import symboldata

def merge (parser, symbolData, stateInfo, modData):
    merger = Merger (parser, symbolData, stateInfo, modData)
    merger.merge ()
    
class Merger:
    def __init__ (self, parser, symbolData, stateInfo, modData):
        self.parser     = parser
        self.symbolData = symbolData
        self.stateInfo  = stateInfo
        self.modData    = modData
        self.accumulatedObjects = {}
        
    def merge (self):
        for file in self.symbolData.files:
            self.symbolData.objectList = []
            self.filename = '%s.sip' % file
            if not self.openFile ():
                continue
                
            print '    merging %s' % self.filename
            debugLevel = 1
#            if file == 'tcpslavebase':
#                debugLevel = 2

            # parse the previous version's sip files here
            self.parser.parse (self.symbolData, self.stateInfo, self.buff, debugLevel, self.filepath)
            
            # put the object list from the parser into a dict
            self.sipIndex = self.symbolData.createSipIndex ()

            # merge with the object list from the h file
            self._merge (file)
                                    
            # add intrafile objects that don't correspond to h file objects
            self.addAccumulatedSipObj (file)
            
            # collect everything from the tail of the file (usually %MappedTypes
            # or %ModuleHeaderCode)
            self.collectTrailingObjects (file)

            # check to see if %ModuleHeaderCode block required and provided            
            if not '%s.sip' % file in  self.modData.noheader and self.moduleHeaderCheck (file, self.symbolData.fileData [file]):
                modHdr = '%%ModuleHeaderCode\n#include <%s.h>\n%%End\n' % file
                block = symboldata.SipBlockObject ('%ModuleHeaderCode', 0, self.stateInfo)
                block.block = modHdr
                self.symbolData.fileData [file][0].blocks.append (block)            

    def openFile (self):
        self.filepath = os.path.join (self.modData.prjdata.prj_prevpath [0], self.modData.modname, self.filename)
        if not os.path.exists (self.filepath):
            return False
            
        m = open (self.filepath, 'r')
        self.buff = m.read ()
        m.close ()
        return True
        
    # future fix!!: it should be possible to scope objects AFTER merging
    # (only those objects new in the new version) since sip file objects
    # should already have correct scope. However there are errors in 
    # the current sip file set (that twine corrects), so until a
    # complete new file set is constructed and compiled, can't copy
    # scope over from sip files.
        
    # the update* methods transfer information from matched objects
    # in the previous version's sip file
    def updateCppAttributes (self, cppObj, sipObj):
        cppObj.attributes.mcNeeded = sipObj.attributes.mcNeeded
        cppObj.attributes.mtNeeded = sipObj.attributes.mtNeeded
        cppObj.attributes.ioNeeded = sipObj.attributes.ioNeeded
        
    def updateCppObj (self, cppObj, sipObj):
        cppObj.blocks = sipObj.blocks
        cppObj.ignore = sipObj.ignore
        cppObj.force  = sipObj.force
        cppObj.sipDoc = sipObj.doc
        
        if cppObj.objectType == 'class':
            self.updateCppClass (cppObj, sipObj)
        elif cppObj.objectType == 'function':
            self.updateCppFunction (cppObj, sipObj)
        elif cppObj.objectType == 'typedef':
            self.updateCppTypedef (cppObj, sipObj)
        elif cppObj.objectType == 'variable':
            self.updateCppVariable (cppObj, sipObj) 
            
    def updateAnnotation (self, cppObj, sipObj):
        cppObj.annotation = sipObj.annotation
        if cppObj.pyName:
            for item in cppObj.annotation:
                if 'PyName' in item:
                    return
            cppObj.annotation.append ('PyName=%s' % cppObj.pyName)
                
    def updateCppClass (self, cppObj, sipObj):
        self.updateAnnotation (cppObj, sipObj)
    
    def updateCppFunction (self, cppObj, sipObj):
        self.updateAnnotation (cppObj, sipObj)
        self.updateCppAttributes (cppObj, sipObj)
        
        if sipObj.cppReturns or sipObj.cppArguments:
            cppObj.cppReturns   = sipObj.cppReturns
            cppObj.cppArguments = sipObj.cppArguments
            cppObj.arguments    = sipObj.arguments
            cppObj.returns      = sipObj.returns
        else:
            for i in range (len (cppObj.returns)):
                self.updateCppAttributes (cppObj.returns [i], sipObj.returns [i])
            for i in range (len (cppObj.arguments)):
                self.updateCppAttributes (cppObj.arguments [i], sipObj.arguments [i])
                # fix!! intersect annotations right now because
                # some are missing from sip files, but some
                # can only be provided by sip files
        #        cppObj.annotation = sipObj.annotation
                for a in sipObj.arguments [i].annotation:
                    if cppObj.arguments [i].annotation and (a not in cppObj.arguments [i].annotation):
                        cppObj.arguments [i].annotation.append (a)
                    else:
                        cppObj.arguments [i].annotation = [a]

    def updateCppTypedef (self, cppObj, sipObj):
        self.updateCppAttributes (cppObj, sipObj)

    def updateCppVariable (self, cppObj, sipObj):
        self.updateAnnotation (cppObj, sipObj)
        self.updateCppAttributes (cppObj, sipObj)
        
    # sort a list of objects by their line number
    def ordinalSort (self, obj1, obj2):
        if obj1.ordinal == obj2.ordinal:
            return 0
        elif obj1.ordinal > obj2.ordinal:
            return 1
        else:
            return -1
    
    # sip objects that weren't matched by h file objects are left over
    # (either from previous versions, or add-ons to the C++ code)
    # they're collected here (by scope - class or namespace)
    # and then inserted in the appropriate place (within the correct
    # scope, at the end) when the entire h file has been processed
    def accumulateSipObjects (self, ordinalStart, ordinalEnd, key = None):
        accumList = []
        sipfile = self.symbolData.sipIndex
        sipObjects = []
        for name in sipfile:
            if sipfile [name]:
                sipObjects += sipfile [name]
        sipObjects.sort (self.ordinalSort)
        
        for i in range (len (sipObjects)):
            obj = sipObjects [i]
            ordinal = obj.ordinal
            if ordinalStart < ordinal < ordinalEnd:
                accumList.append (obj)
                if obj.objectType not in ['sipBlock', 'mappedtype', 'exception']:
                    obj.version.versionHigh = self.modData.version
                sipfile.delete (obj)
            elif ordinal >= ordinalEnd:
                break
            
        if key and accumList:
            self.accumulatedObjects [key] = accumList
        return accumList
        
    # this is where unmatched sip objects get inserted
    def addAccumulatedSipObj (self, file):
        if len (self.accumulatedObjects) > 0:
            cppfile = self.symbolData.fileData [file]
            keys = self.accumulatedObjects.keys ()
            keys.sort ()
            keys.reverse ()
            for key in keys:
                cppfile = cppfile [0:key] + self.accumulatedObjects [key] + cppfile [key:]
            self.accumulatedObjects = {}
            self.symbolData.fileData [file] = cppfile
            

    def _merge (self, file):
        cppfile = self.symbolData.fileData [file]
        sipfile = self.symbolData.sipIndex
        ordinalStack = []
        skipSip = 0
        skipCpp = 0
        
        # for each h file (cppObj) object, try to find a matching sip file
        # object - ignored classes are handled here
        length = len (cppfile)
        for i in range (length):
            cppObj = cppfile [i]
            
            # test for and skip empty namespace blocks
            if (i < length - 1)\
                and cppObj.objectType == 'namespace'\
                and cppfile [i + 1].objectType == 'endnamespace':
                cppObj.skip = True
                cppfile [i + 1].skip = True
                continue
                
            matched = False
            if not (skipSip or skipCpp):
                sipObjList = sipfile.getMatchingObjects (cppObj.name)
                if not sipObjList:
                    cppObj.version.versionLow = self.modData.version
                    if cppObj.objectType in ['namespace', 'class']:
                        ordinalStack.append (cppObj.ordinal)
                    continue
                    
                for sipObj in sipObjList:
                    if sipObj.version.versionHigh:
                        continue
                        
                    elif cppObj.objectType == 'class' and sipObj.objectType == 'class' and sipObj.ignore:
                        cppObj.ignore = True
                        skipCpp = 1
                        matched = True
                        sipfile.delete (sipObj)
                                                    
                    elif sipObj == cppObj:
                        matched = True
                        if sipObj.objectType in ['namespace', 'class']:
                            ordinalStack.append (sipObj.ordinal)
                        elif sipObj.objectType in ['endnamespace', 'endclass'] and not cppObj.skip:                            
                            ordinalStart = ordinalStack.pop ()
                            ordinalEnd   = sipObj.ordinal
                            self.accumulateSipObjects (ordinalStart, ordinalEnd, i)
                            
                        self.updateCppObj (cppObj, sipObj)
                        sipfile.delete (sipObj)
                        break
                        
                    else:
                        if cppObj.objectType in ['namespace', 'class']:
                            ordinalStack.append (cppObj.ordinal)
                    
            elif skipCpp:
                cppObj.ignore = True
            
            if matched:
                continue
            
            if not skipCpp:
                cppObj.version.versionLow = self.modData.version
            
            if cppObj.objectType == 'class':
                skipSip += 1
#                if skipSip: skipSip += 1
#                if skipCpp: skipCpp += 1
            
            elif cppObj.objectType in ['endclass', 'endnamespace']:
                if skipSip: skipSip -= 1
                if skipCpp: skipCpp -= 1

    # pick up anything unmatched in the sip file that hasn't 
    # already been collected - should be only objects with
    # global scope
    def collectTrailingObjects (self, file):
        accumList = self.accumulateSipObjects (0, 10000)
        if accumList:
            self.symbolData.fileData [file] += accumList

    def moduleHeaderCheck (self, filename, objList):
        hasModHdr  = False
        inClass    = 0
        needModHdr = False
        include = '#include <%s.h>' % filename        
        
        for obj in objList:
            if obj.objectType in ['function', 'typedef', 'enum', 'variable'] and not inClass:
                needModHdr = True
            elif obj.objectType == 'class':
                inClass += 1
            elif obj.objectType == 'endclass':
                inClass -= 1
            
            for block in obj.blocks:
                if block.name == '%ModuleHeaderCode' and include in block.block:
                    hasModHdr = True
                    
        return needModHdr and not hasModHdr
        
            
