#     Copyright 2007-8 Jim Bublitz <jbublitz@nwinternet.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

import sys, os, os.path
from sipparser import SipParser
from symboldata import Data
from stateInfo import StateInfo
from cppparser import CppParser
from pplexer import preprocess
from twineConfig import scopeObjects, mergePrevious, generateOutput, mergePrevious, subClass


def error (text):
    print
    print text
    print
    sys.exit (-1)

class TopLevelParser:
    """
    This class processes top level files for either modules to be
    generated, or for imported modules. It runs the entire process
    from start to finish
    """
    def __init__ (self, imported = False, symbolData = None):
        self.imported   = imported
        
        if not symbolData:
            self.symbolData = Data ()
        else:
            self.symbolData = symbolData

        self.stateInfo =  StateInfo ()        
        self.sipParser  = SipParser ()
        self.hParser    = CppParser ()
        self.modules    = []
        self.modulePath = ''

    def processModule (self, filename, modData):
        """
        Processes one module and the module's imports
        """
        self.filename = filename
        self.modData  = modData
        self.initState (modData)
        self.symbolData.hierarchy.clearCurrentMod ()
        
        try:
            self.parse ()
        except:
            print
            print "Error parsing top level file %s" % self.filename
            print
            raise

        self.processDirectives ()
        if not self.imported:
            self.output ()
            self.symbolData.clear ()

    def initState (self, modData):
        """
        Set up the module level StateInfo and SymbolData objects
        """
        self.stateInfo.clear ()
        self.stateInfo.module   = modData.modname
        self.stateInfo.filename = '%smod.sip.in' % modData.modname
        self.symbolData.objectList = []
    
    def parse (self):
        """
        Parse the top level file (new module or its imports)
        Each directive is an entry in the module level
        symbolData.objectList
        """
        print("Parsing sip file: "+self.filename)
        m = open (self.filename, 'r')        
        self.buff = m.read ()
        m.close ()
        self.sipParser.parse (self.symbolData, self.stateInfo, self.buff, filename=self.filename)

    def processDirectives (self):
        """
        Do the imports and includes indicated in the top level file
        """
        name = ''
        for obj in self.symbolData.objectList:

            if obj.objectType != 'sipDirective':
                continue
            if obj.name == 'Module':
                name = obj.argument.split (',') [0].strip ().split ('.') [-1]
                            
            elif obj.name == 'Timeline':
                pass

            elif obj.name == 'Platforms':
                pass

            elif obj.name == 'Features':
                pass
            
            elif obj.name == 'Import':
                self.parseImportModule (obj.argument)
                self.symbolData.hierarchy.clearCurrentMod ()
                
            elif obj.name == 'Include':
                self.parseIncludedFile (obj.argument, name)

    def computeModName (self, moduleName):
        # the name of the module currently being processed
        if '/' in moduleName:
            modName = os.path.split (moduleName) [1]
        else:
            modName = moduleName
        if modName.endswith ('mod.sip'):
            modName = modName [:-7]
            
        return modName
    
            
    def findImportedModule (self, moduleName):
        # search the list of paths from the project file
        for path in self.modData.importPath:
            modulePath = os.path.join (path, moduleName)
            if os.path.exists (modulePath):
                return modulePath

        return None                      

    def parseImportModule (self, moduleName):
        """
        Import a module
        """
        
        modName = self.computeModName (moduleName)
        
        if modName in self.modules:
            return

        modulePath = self.findImportedModule (moduleName)
        if modulePath == None:
            error ("module %s not found" % moduleName)
                    
        importTopLevel            = TopLevelParser (True, self.symbolData)
        importTopLevel.modules    = self.modules
        importTopLevel.modulePath = os.path.dirname (modulePath)
        importTopLevel.processModule (modulePath, self.modData)
        self.modules.append (modName)

    def findIncludedFile (self, filepath, paths):
        # search the paths for h files defined in the project module
        # and by the file selection mechanism
        for path in paths:
            file = os.path.join (path, filepath)
            if os.path.exists (file):
                return file
        return None
                        
    def parseIncludedFile (self, filepath, moduleName = ''):
        """
        Parse an h file or sip file
        """
        if '/' in filepath:
            filename = os.path.split (filepath) [1]
        else:
            filename = filepath
        filename, fileext = os.path.splitext (filename)
        if not self.imported and not filepath in self.modData.noheader:
            fileext = '.h'
            
        fileStateInfo  = StateInfo ()
        fileSymbolData = Data ()
        fileStateInfo.module   = self.modData.modname
        fileStateInfo.filename = filepath
        if filepath in self.modData.noheader:
            file_name = self.findIncludedFile (os.path.join (moduleName, filename + fileext), self.modData.prevpath)
        elif fileext == '.h':
            file_name = self.findIncludedFile (filename + fileext, self.modData.substPath + self.modData.sourcePath)
        elif fileext == '.sip':
            file_name = self.findIncludedFile (filepath, [self.modulePath])
            
        if file_name is None:
            print("Unable to handle included file: "+filepath)
            return
            
        m = open (file_name, 'r')
        self.buff = m.read ()
        m.close ()
        
        # determine which parser to use and invoke it
        if fileext == '.h':
            print('Parsing included .h file %sh (%s)' % (filepath [:-3],file_name))
            debug = 1
##            if filename == 'acadapter':
##                debug = 2
            self.hParser.parse (fileSymbolData, fileStateInfo, preprocess (self.buff, self.modData.values, self.modData.defines), debug)
            if debug == 2:
                sys.exit (-1)
        elif fileext == '.sip':
            print("Parsing included .sip file %s (%s)" % (filepath,file_name))
            self.sipParser.parse (fileSymbolData, fileStateInfo, self.buff, 1, filename=file_name)
            
        # take the info from parsing and store it for future use
        if self.imported:
            self.symbolData.addFile (filename, fileSymbolData.objectList, None, self.imported, self.modData.imports [moduleName])
        else:
            self.symbolData.addFile (filename, fileSymbolData.objectList, self.modData)

    def output (self):
        # produce the output after some more processing steps
        scopeObjects (StateInfo (), self.symbolData, self.modData)
        mergePrevious (self.sipParser, self.symbolData, self.stateInfo, self.modData)
        generateOutput (self.symbolData, self.modData, StateInfo ())
        subClass (self.symbolData, self.modData)
