#!/usr/bin/perl -w

# Copyright 2008 Tobias Hunger <tobias.hunger@basyskom.de>
#
# This code is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file takes a telepathy spec file and converts it to XML suitable for
# qdbusxml2cpp.
#
# This script must be run from the scripts/helpers directory of the
# Telepathy-Qt checkout!
#
# usage: $0 spec-file-name output-file-name

# Required binaries:
#  tempfile
#  xsltproc

use strict;

use lib "helpers";
use tp;

# ------------------ do not edit below this line -------------------

# Process CLI:
my $basename = shift;
my $input_dir = shift;
my $qdbus_dir = shift;

die "Usage: $0 basename spec-dir qdbus-xml-dir\n"
    unless $qdbus_dir;

print "\n\n$basename:\n";

# ------------------------------------------------------------------
# Generate QDBus XML file:
# ------------------------------------------------------------------

sub generate_qdbusxml
{
    my $basename = shift;
    my $input_file = shift;
    my $qdbus_dir = shift;

    die "Missing parameters" unless $qdbus_dir;

    my $qdbus_file = "$qdbus_dir/$basename.xml";


    # Use XSLTproc to trim down tags in the XML and store it into a
    # temporary file (so that we can use perl to patch up the output
    # later):

    my $tmp_file = tp::filename($basename);
    $tmp_file .= ".xml.tmp";

    `xsltproc "$tp::qdbusxml_template" "$input_file" > "$tmp_file"`;
    if ($? != 0)
    { die "XSLTproc failed with return code $? while generating QDBusXML.\n"; }

    print "    ($qdbus_file)\n";

    # Postprocess the generated file using perl:
    open(TMP, "<$tmp_file") or
        die "Failed to open temporary file \"$tmp_file\" (RO): $!.\n";
    open(OUT, ">$qdbus_file") or
        die "Failed to open output file \"$qdbus_file\" (WO): $!\n";

    my $in_count = -1;
    my $out_count = -1;
    my $type = "";
    my $tptype = "";
    my $indent = "";

    my $direction = "";
    my $current_direction = "";

    while(<TMP>)
    {
        my $line = "$_";
        chomp $line;

        foreach my $i ( @tp::keywords )
        {
            $line =~ s/name=\"$i\"/name=\"param_$i\"/g
        }

        # <method>/<signal>
        if ($line =~ /<(method|signal)/i)
        {
            $current_direction = "In";
            $direction = "";
            $in_count = -1;
            $out_count = -1;
        }

        # direction
        if ($line =~ /direction="in"/)
        { $current_direction = "In"; }
        if ($line =~ /direction="out"/)
        { $current_direction = "Out"; }

        # type:
        if ($line =~ /type="([^\"]*)"/)
        {
            if ($current_direction eq "In") { $in_count += 1; }
            elsif ($current_direction eq "Out") { $out_count += 1; }
            $type = $1;
        }

        # tp:type
        if ($line =~ /tp:type="([^\"]*)"/)
        {
            my $tmp = "$1";
            $tmp =~ s/_//g;
            $tptype = "org::freedesktop::Telepathy::$tmp";
            $tptype =~ s/(.*)\[\]$/QList< $1 >/;

            $indent = $line;
            $indent =~ /^(\s*)/;
            $indent = $1;

            $tptype = $tp::known_types{$type} if exists $tp::known_types{$type};

            $tptype =~ s/</&lt;/;
            $tptype =~ s/>/&gt;/;

            # Use default direction if none is explicitly given:
            if ($current_direction eq "Out") { $direction = "Out$out_count"; }
            else { $direction = "In$in_count"; }

            print OUT "$indent<annotation name=\"com.trolltech.QtDBus.QtTypeName.$direction\"\n";
            print OUT "$indent    value=\"$tptype\"/>\n";

            print "$tptype\t$type\n";
        }

        $line =~ s/\s*tp:type="[^\"]*"\s*//;

        # </method>/</signal>

        print OUT "$line\n";
    }

    close OUT;
    close TMP;

    # Clean up empty qdbus files:
    `grep "<node" "$tmp_file" > /dev/null || rm "$qdbus_file"`;

    # Clean up temporary file:
    `rm "$tmp_file"` unless $tp::keep_tmp_files;
}

# ------------------------------------------------------------------
# Main:
# ------------------------------------------------------------------

my $input_file = "$input_dir/$basename.xml";

print "    ... QDBus XML\n";
generate_qdbusxml($basename, $input_file, $qdbus_dir);
print "    ... Done.\n";

exit 0
