#!/usr/bin/perl -w

# Copyright 2008 Tobias Hunger <tobias.hunger@basyskom.de>
#
# This code is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file takes a telepathy spec file and converts it to XML suitable for
# qdbusxml2cpp.
#
# This script must be run from the scripts/helpers directory of the
# Telepathy-Qt checkout!
#
# usage: $0 spec-file-name output-file-name

# Required binaries:
#  tempfile
#  xsltproc

use strict;

use lib "helpers";
use tp;

# ------------------ do not edit below this line -------------------

# Process CLI:
my $basename = shift;
my $input_dir = shift;
my $header_dir = shift;
my $sources_dir = shift;

die "Usage: $0 basename spec-dir header-dir sources_dir\n"
    unless $sources_dir;

print "\n\n$basename:\n";

# ------------------------------------------------------------------
# Generate Header file:
# ------------------------------------------------------------------

sub get_type
{
    my $tp_type = shift;
    my $dbus_type = shift;
    my $types = shift;

    $tp_type =~ s/_//g;

    if ($tp_type =~ /([a-zA-Z0-9]*)\[\]$/)
    { $tp_type = "QList< $1 >"; }

    if (exists $tp::known_types{$dbus_type} &&
        !exists $types->{$tp_type})
    {
        print "    Replacing \"$tp_type\" with \"".
              $tp::known_types{$dbus_type}."\"\n";
        $tp_type = $tp::known_types{$dbus_type};
    }
    return $tp_type;
}

sub dump_types
{
    my $types = shift;
    my $value = 0; # calculate max. value for flags.

    # Remember known types:
    my %defined_types = ();
    foreach my $i ( keys %$types )
    {
        # remember valid types only:
        $defined_types{$i} = 1 if $types->{$i}->{VALID};
        # get rid of the invalid ones:
        delete $types->{$i} unless $types->{$i}->{VALID};
    }
    # Find types that we can not resolve:
    my %unresolvable_types = ();
    # ... first find all required types:
    foreach my $t (keys %$types)
    {
        foreach my $st (keys %{$types->{$t}->{REQ}})
        {
            $unresolvable_types{$st} = 1;
            print "Required by \"$t\": \"$st\".\n";
        }
    }
    # ... substract defined types now:
    foreach my $t (keys %$types)
    { delete $unresolvable_types{$t} if exists $unresolvable_types{$t}; }

    if (scalar keys %unresolvable_types)
    {
        print OUT "// The following types are not defined inside the current XML file:\n";
        foreach my $t (sort keys %unresolvable_types)
        {
            print "Unresolved type: \"$t\".\n";
            print OUT "//     $t\n";
        }
        print OUT "// You have to make sure that they are either defined in generic-types\n";
        print OUT "// or are typedefs to simple data types!\n";
    }

    # Remove unresolvable types from all REQ fields:
    foreach my $t (keys %$types)
    {
        foreach my $st (keys %unresolvable_types)
        { delete $types->{$t}{REQ}{$st} if exists $types->{$t}{REQ}{$st}; }
    }

    # Only resolveable types are left, order them:
    while (scalar keys %$types)
    {
        my $curr_type = "";
        # find a type that has no prerequisite:
        foreach my $t (keys %$types)
        {
            if (not scalar keys %{$types->{$t}{REQ}})
            {
                # This type has no prerequesits:
                $curr_type = $t;
                last;
            }
        }
        if ($curr_type eq "") { die "Unresolved type found!\n"; }

        print "Current type: \"$curr_type\"...\n";

        # remove the current type from all other types:
        foreach my $t (keys %$types)
        {
            next if $t eq $curr_type;
            delete $types->{$t}{REQ}{$curr_type}
                if exists $types->{$t}{REQ}{$curr_type};
        }

        # output the current type:
        my @lines = split(/\n/, $types->{$curr_type}{TEXT});
        my $text = "";
        my $next_type = "";
        my $next_type2 = "";
        my $next_member = "";
        my $next_name = "";
        foreach my $line ( @lines )
        {
            if ($line =~ /^\s*##TYPE_PLACEHOLDER: (.*)%(.*)#\s*$/)
            {
                $next_type = get_type("$1", "$2", \%defined_types);
                next;
            }
            if ($line =~ /^\s*##TYPE2_PLACEHOLDER: (.*)%(.*)#\s*$/ )
            {
                $next_type2 = get_type("$1", "$2", \%defined_types);
                next;
            }
            if ($line =~ /^\s*##MEMBER_PLACEHOLDER: (.*)#\s*$/)
            {
                $next_member = "$1";
                $next_member =~ s/_//g;
                $next_member = lcfirst($next_member);
                next;
            }
            if ($line =~ /^\s*##NAME_PLACEHOLDER: (.*)#\s*$/)
            {
                $next_name = "$1";
                $next_name =~ s/_//g;
                next;
            }

            $line =~ s/NAME_PLACEHOLDER/$next_name/g;
            $line =~ s/TYPE_PLACEHOLDER/$next_type/g;
            $line =~ s/TYPE2_PLACEHOLDER/$next_type2/g;
            $line =~ s/MEMBER_PLACEHOLDER/$next_member/g;

            $text .= "$line\n";
        }

        # remove last ',' in enums:
        if ($text =~ /\benum\b/)
        {
            my @lines = split /\n/, $text;
            $lines[-3] =~ /(\d+)/;
            $value = $1;
            $value = $value * 2;
            $lines[-2] =~ s/FLAG_MAX_PLACEHOLDER/$value/;
            $text = join "\n", @lines;
            $text .= "\n"; # add the last newline that got removed by split/join
        }
        print OUT "$text\n";

        # delete the current type:
        delete $types->{$curr_type};
    }
}

sub generate_typeheaders
{
    my $basename = shift;
    my $input_file = shift;
    my $header_dir = shift;

    die "Missing parameters!" unless $header_dir;

    $basename =~ s/[^a-zA-Z0-9_]/_/g;
    if ($basename !~ /types$/) { $basename .= "_Types"; }

    my $actual_header = lc($basename . "_gen.h");
    my $pretty_header = $basename;
    my $manual_header = lc($basename . ".h");

    my $actual_header_guard = "QTTELEPATHY_".uc($basename)."_GENERATED_H";
    my $manual_header_guard = "QTTELEPATHY_".uc($basename)."_H";

    my $full_path_actual_header = "$header_dir/$actual_header";
    my $full_path_pretty_header = "$header_dir/$pretty_header";
    my $full_path_manual_header = "$header_dir/$manual_header";

    # Use XSLTproc to generate a raw version of the header and store
    # it into a temporary file (so that we can use perl to patch up the output
    # later):

    my $tmp_file = tp::filename($basename);
    $tmp_file .= ".h.tmp";

    `xsltproc "$tp::header_template" "$input_file" > "$tmp_file"`;
    if ($? != 0)
    { die "XSLTproc failed with return code $? while generating header.\n"; }

    # Postprocess the generated file using perl:
    my $header_is_empty = 1;

    open(TMP, "<$tmp_file") or
        die "Failed to open temporary file \"$tmp_file\" (RO): $!.\n";
    open(OUT, ">$full_path_actual_header") or
        die "Failed to open output file \"$full_path_actual_header\" (WO): $!\n";

    my %type_list;
    my $quote = 0;
    my $current_type = "";
    my $current_dbus_type = "";
    my $metatype_info = "";
    my $marshalling = "";
    my $skip_type = 0;

    # Telepathy has no ordering in the types it defines. We have to reorder them
    # ourselves:-(
    while(<TMP>)
    {
        my $line = $_;
        chomp $line;

        $line =~ s/HEADER_GUARD/$actual_header_guard/;

        # handle quoted area:
        if ($line =~ /^#START-OF-HEADER#$/)
        {
            $quote = 1;
            next;
        }
        if ($line =~ /^#START-OF-FOOTER#$/)
        {
            $quote = 1;
            dump_types(\%type_list);
            if ($basename eq "generic-types")
            {
                print OUT "// register types with QtDBus.\n";
                print OUT "void registerTypes();\n";
            }
            next;
        }
        if ($line =~ /^(#END-OF-HEADER#|#END-OF-FOOTER#)$/)
        {
            $quote = 0;
            next;
        }
        if ($line =~ /^\s*#GENERATED-CODE#\s*$/)
        {
            print OUT "$metatype_info\n";
            print OUT "$marshalling\n";
            next;
        }
        if ($quote)
        {
            print OUT "$line\n";
            next;
        }

        if ($line =~ /^\s*##START-OF-TYPE: (.*)%(.*)#\s*$/)
        {
            print "Start: $line\n";
            die "In type '$current_type' when running into START-OF-TYPE!"
                unless $current_type eq "";
            $current_type = "$1";
            $current_dbus_type = "$2";
            $current_type =~ s/_//g;
            if (exists $tp::known_types{$current_dbus_type})
            {
                $skip_type = 1;
                $type_list{$current_type} = ();
                $type_list{$current_type}{VALID} = 0;
                print "     SKIPPING $current_type!\n";
            }
            else
            {
                $skip_type = 0;
                $header_is_empty = 0;
                $type_list{$current_type} = ();
                $type_list{$current_type}{TEXT} = "";
                $type_list{$current_type}{REQ} = ();
                $type_list{$current_type}{VALID} = 1;
            }
            next;
        }
        if ($line =~ /^\s*##END-OF-TYPE: (.*)%(.*)#\s*$/)
        {
            print "End: $line\n";
            my $end_of_type = "$1";
            $end_of_type =~ s/_//g;
            die "Type mismatch!" unless $current_type eq $end_of_type;
            die "Not in a type when running into END-OF-TYPE!" if $current_type eq "";
            $current_type = "";

            next if $skip_type;

            # Make sure we have no [] lists anymore:
            foreach my $t (keys %{$type_list{$current_type}{REQ}})
            { $type_list{$current_type}{TEXT} =~ s/$t\[\]/QList< $t >/g; }
            next;
        }

        if ($skip_type)
        {
            print ".... $line\n";
            next;
        }

        if ($line =~ /^\s*##REQUIRE-QT-METATYPE: (.*)#\s*$/)
        {
            my $magic = "$1";
            $magic =~ s/_//g;
            if ($magic =~ /^QList\s*<\s*([A-Za-z0-9_]*)\s*>$/)
            { $magic = "QList< org::freedesktop::Telepathy::$1 >"; }
            else
            { $magic = "org::freedesktop::Telepathy::$magic"; }
            $metatype_info .= "Q_DECLARE_METATYPE($magic)\n";
            next;
        }
        if ($line =~ /^\s*##REQUIRE-MARSHALLING: (.*)#\s*$/)
        {
            my $magic = "$1";
            $magic =~ s/_//g;
            if ($magic =~ /^QList\s*<\s*(.*)\s*>$/)
            { $magic = "QList< org::freedesktop::Telepathy::$1 >"; }
            else
            { $magic = "org::freedesktop::Telepathy::$magic"; }
            $marshalling .= "QTTELEPATHY_COMMON_EXPORT const QDBusArgument & operator >> (const QDBusArgument &, $magic &);\n";
            $marshalling .= "QTTELEPATHY_COMMON_EXPORT QDBusArgument & operator << (QDBusArgument &, const $magic &);\n\n";
            next;
        }

        if ($current_type)
        {
            next if $line =~ /^\s*$/;
            if ($line =~ /^\s*##REQ: (.*)#\s*$/)
            {
                my $req_type = $1;
                $req_type =~ s/_//g;
                $req_type = substr($req_type, 0, length($req_type) - 2)
                    if $req_type =~ /\[\]$/;
                $type_list{$current_type}{REQ}{$req_type} = 1;
                next;
            }
            else
            {
                $type_list{$current_type}{TEXT} .= "$line\n";
                next
            }
        }
    }
    print OUT "\n";

    close OUT;
    close TMP;

    if ($header_is_empty)
    {
        print "  HEADER IS EMPTY!\n";
        # Clean up empty header files:
        `rm "$full_path_actual_header"`;
    }
    else
    {
        # Create pretty header file:
        `echo "#include <QtTelepathy/Common/$manual_header>" > "$full_path_pretty_header"`;

        # Create private header if if does not exist yet:
        unless (-e "$full_path_manual_header")
        {
            `echo "/* Add your code here! */" >  "$full_path_manual_header"`;
            `echo >> "$full_path_manual_header"`;
            `echo "#ifndef $manual_header_guard" >> "$full_path_manual_header"`;
            `echo "#define $manual_header_guard" >> "$full_path_manual_header"`;
            `echo >> "$full_path_manual_header"`;
            `echo "#include <QtCore/QString>" >> "$full_path_manual_header"`;
            `echo "#include <QtCore/QVariant>" >> "$full_path_manual_header"`;
            `echo "#include <QtDBus/QDBusObjectPath>" >> "$full_path_manual_header"`;
            `echo "#include <QtDBus/QDBusSignature>" >> "$full_path_manual_header"`;
            `echo "#include <QtDBus/QDBusArgument>" >> "$full_path_manual_header"`;
            `echo >> "$full_path_manual_header"`;
            `echo "#include <QtTelepathy/Common/$actual_header>" >> "$full_path_manual_header"`;
            `echo >> "$full_path_manual_header"`;
            `echo "#endif // header guard" >> "$full_path_manual_header"`;
            `echo >> "$full_path_manual_header"`;
        }
        else
        { print "  MANUAL HEADER FILE EXISTS!\n\n\n"; }
    }

    # Clean up temporary file:
    `rm "$tmp_file"` unless $tp::keep_tmp_files;
}

# ------------------------------------------------------------------
# Type Sources:
# ------------------------------------------------------------------

sub generate_typesources
{
    my $basename = shift;
    my $input_file = shift;
    my $sources_dir = shift;

    die "Missing types!" unless $sources_dir;

    $basename =~ s/[^a-zA-Z0-9_]/_/g;
    if ($basename !~ /types$/) { $basename .= "_Types"; }

    my $actual_source = lc($basename . "_gen.cpp");
    my $full_path_actual_source = "$sources_dir/$actual_source";

    # Use XSLTproc to generate a raw version of the header and store
    # it into a temporary file (so that we can use perl to patch up the output
    # later):

    my $tmp_file = tp::filename($basename);
    $tmp_file .= ".cpp.tmp";

    `xsltproc "$tp::sources_template" "$input_file" > "$tmp_file"`;
    if ($? != 0)
    { die "XSLTproc failed with return code $? while generating header.\n"; }

    # Postprocess the generated file using perl:
    my $source_is_empty = 1;

    open(TMP, "<$tmp_file") or
        die "Failed to open temporary file \"$tmp_file\" (RO): $!.\n";
    open(OUT, ">$full_path_actual_source") or
        die "Failed to open output file \"$full_path_actual_source\" (WO): $!\n";

    my $next_member = "";
    my $next_type = "";
    my $current_type = "";
    my $current_dbus_type = "";
    my $skip_type = 0;

    while(<TMP>)
    {
        my $line = $_;
        chomp $line;

        $line =~ s/HEADER_FILE/$basename/g;

        $source_is_empty = 0 if ($line =~ /\boperator\b/);

        if ($line =~ /^\s*##MEMBER_PLACEHOLDER: (.*)#\s*$/)
        {
            $next_member = "$1";
            $next_member =~ s/_//g;
            $next_member = lcfirst($next_member);
            next;
        }
        if ($line =~ /^\s*##TYPE_PLACEHOLDER: (.*)%(.*)#\s*$/)
        {
            $next_type = "$1";
            $next_type =~ s/_//g;
            $next_type = "org::freedesktop::Telepathy::$next_type";
            next;
        }
        if ($line =~ /^\s*##START-OF-TYPE: (.*)%(.*)#\s*$/)
        {
            die "In type '$current_type' when running into START-OF-TYPE!"
                unless $current_type eq "";
            $current_type = "$1";
            $current_dbus_type = "$2";
            $current_type =~ s/_//g;
            if (exists $tp::known_types{$current_dbus_type})
            {
                $skip_type = 1;
                print "     SKIPPING!\n";
            }
            else { $skip_type = 0; }
            next;
        }
        if ($line =~ /^\s*##END-OF-TYPE: (.*)%(.*)#\s*$/)
        {
            die "Not in a type when running into END-OF-TYPE!" if $current_type eq "";
            $current_type = "";
            $next_member = "";
            next;
        }

        if ($current_type && !$skip_type)
        {
            next if $line =~ /^\s*$/;
            $line =~ s/TYPE_PLACEHOLDER/$next_type/;
            $line =~ s/MEMBER_PLACEHOLDER/$next_member/;
        }

        print OUT "$line\n";
    }

    close OUT;
    close TMP;

    if ($source_is_empty)
    {
        print "  SOURCE IS EMPTY!\n";
        # Clean up empty header files:
        `rm "$full_path_actual_source"`;
    }

    # Clean up temporary file:
    `rm "$tmp_file"` unless $tp::keep_tmp_files;
}

# ------------------------------------------------------------------
# Main:
# ------------------------------------------------------------------

my $input_file = "$input_dir/$basename.xml";

print "    ... type headers\n";
generate_typeheaders($basename, $input_file, $header_dir);
print "    ... type sources\n";
generate_typesources($basename, $input_file, $sources_dir);
print "    ... Done.\n";

exit 0
