/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include <QDebug>

#include "QtTapioca/UserContact"
#include "QtTapioca/Handle"
#include "QtTapioca/Avatar"

#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusConnectionInterface>
#include <QtTelepathy/Client/Connection>

namespace QtTapioca {

class UserContactPrivate {
public:
    UserContactPrivate(org::freedesktop::Telepathy::Connection *telepathyConn)
        :bus(QDBusConnection::sessionBus())
    {
        this->telepathyConn = telepathyConn;
    }
    ~UserContactPrivate()
    {
    }

    QDBusConnection bus;
    org::freedesktop::Telepathy::Connection *telepathyConn;
};

}

using namespace QtTapioca;

UserContact::UserContact(QtTapioca::Connection *connection,
                         org::freedesktop::Telepathy::Connection *telepathyConn,
                           org::freedesktop::Telepathy::ConnectionInterfaceAvatars  *iAvatar,
                           org::freedesktop::Telepathy::ConnectionInterfacePresence *iPresence,
                           org::freedesktop::Telepathy::ConnectionInterfaceAliasing *iAliasing,
                           org::freedesktop::Telepathy::ConnectionInterfaceCapabilities *iCapabilities,
                           Handle *handle,
                           QObject *parent)
    : ContactBase(connection, telepathyConn, iAvatar, iPresence, iAliasing, iCapabilities, handle, parent),
      d(new UserContactPrivate(telepathyConn))
{
}

UserContact::~UserContact()
{
    delete d;
}

bool UserContact::setAvatar(Avatar *avatar)
{
    if ((!avatar) || (!telepathyIAvatar))
        return false;

    QDBusReply<QString> reply = telepathyIAvatar->SetAvatar(avatar->data(), avatar->mimeType());

    if (!reply.isValid()) {
        qDebug() << "error on setting the avatar:" << reply.error().message();
        return false;
    }

    avatar->setToken(reply.value());

    return true;
}

bool UserContact::setPresence(const QtTapioca::PresenceState & state)
{ return setPresenceWithMessage(state, presenceMessage()); }

bool UserContact::setPresenceMessage(const QString &message)
{ return setPresenceWithMessage(presence(), message); }

bool UserContact::setPresenceParameters(const QVariantMap &params)
{ return setPresenceWithParameters(presence(), params); }

bool UserContact::setPresenceWithMessage(const QtTapioca::PresenceState & state, const QString &message)
{
    QVariantMap new_params(m_presenceParams);
    new_params["message"] = message;

    return setPresenceWithParameters(state, new_params);
}

bool UserContact::setPresenceWithParameters(const QtTapioca::PresenceState & state,
                                            const QVariantMap & params)
{
    if (!telepathyIPresence)
        return false;

    org::freedesktop::Telepathy::MultipleStatusMap presenceStatus;

    presenceStatus.insert(state.name(), params);

    QDBusReply<void> reply = telepathyIPresence->SetStatus(presenceStatus);

    if (!reply.isValid())
    {
        qDebug() << "error setting presence status:" << reply.error().message();
        return false;
    }
    return true;
}

bool UserContact::setAlias(const QString &alias)
{
    if (!telepathyIAliasing)
        return false;

    QMap<uint, QString> map;
    map.insert(handle()->id(), alias);

    QDBusReply<void> reply = telepathyIAliasing->SetAliases(map);

    if (!reply.isValid()) {
        qDebug() << "error on setting alias:" << reply.error().message();
        return false;
    }

    return true;
}

bool UserContact::setCapabilities(QList<ContactBase::Capability> caps)
{
    if (!telepathyICapabilities)
        return false;

    QStringList remove;
    QList<org::freedesktop::Telepathy::CapabilityPair> add;
    QDBusReply<QList<org::freedesktop::Telepathy::CapabilityPair> > reply;

    /* Reset all caps */
    remove << "org.freedesktop.Telepathy.Channel.Type.Text";
    remove << "org.freedesktop.Telepathy.Channel.Type.StreamedMedia";

    reply = telepathyICapabilities->AdvertiseCapabilities(add, remove);

    if (!reply.isValid()) {
        qDebug() << "error on advertise capabilities:" << reply.error().message();
        return false;
    }

    //Check if Text is supported
    if (caps.contains(ContactBase::Text)) {
        org::freedesktop::Telepathy::CapabilityPair capInfo;
        capInfo.channelType =  "org.freedesktop.Telepathy.Channel.Type.Text";
        add << capInfo;
    }

    //Check if Audio and video are supported
    if ((caps.contains(ContactBase::Audio)) ||
        (caps.contains(ContactBase::Video))) {
        org::freedesktop::Telepathy::CapabilityPair capInfo;
        capInfo.channelType =  "org.freedesktop.Telepathy.Channel.Type.StreamedMedia";

        if (caps.contains(ContactBase::Audio))
            capInfo.typeSpecificFlags |= 1;
        if (caps.contains(ContactBase::Video))
            capInfo.typeSpecificFlags |= 2;

        add << capInfo;
    }

    remove.clear();

    reply = telepathyICapabilities->AdvertiseCapabilities(add, remove);

    if (!reply.isValid()) {
        qDebug() << "error on advertise capabilities:" << reply.error().message();
        return false;
    }

    return true;
}
