/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include "QtTapioca/ConnectionManagerFactory"
#include "QtTapioca/ConnectionManager"
#include "QtTapioca/PresenceState"

#include <QtCore/QDebug>
#include <QtCore/QDir>
#include <QtDBus/QDBusMetaType>
#include <QtTelepathy/Common/Types>
#include <QtTelepathy/Common/Connection_Interface_Presence_Types>
#include <QtTelepathy/Common/Connection_Interface_Capabilities_Types>
#include <QtTelepathy/Common/Connection_Interface_Aliasing_Types>
#include <QtTelepathy/Common/Connection_Types>
#include <QtTelepathy/Common/Channel_Type_Text_Types>
#include <QtTelepathy/Common/Channel_Type_Streamed_Media_Types>

using namespace QtTapioca;

ConnectionManagerFactory *ConnectionManagerFactory::s_self = 0;

ConnectionManagerFactory *ConnectionManagerFactory::self()
{
    if (!s_self)
        s_self = new ConnectionManagerFactory();
    return s_self;
}

ConnectionManagerFactory::ConnectionManagerFactory()
    : QObject(0),
      d(0)
{
    registerTypes();
    load();
}

ConnectionManagerFactory::~ConnectionManagerFactory()
{
    s_self = 0;

    ConnectionManager *cm;
    foreach (cm, m_cms) {
        delete cm;
    }
}

const QList<ConnectionManager*> &ConnectionManagerFactory::getAllConnectionManagers() const
{
    return m_cms;
}

QList<ConnectionManager*> ConnectionManagerFactory::getConnectionManagers(const QString &protocol) const
{
    QList<ConnectionManager*> found;
    ConnectionManager *cm;
    foreach (cm, m_cms) {
        if (cm->supports(protocol)) {
            found.append(cm);
        }
    }
    return found;
}

ConnectionManager *ConnectionManagerFactory::getConnectionManager(const QString &protocol) const
{
    ConnectionManager *cm;
    foreach (cm, m_cms) {
        if (cm->supports(protocol)) {
            return cm;
        }
    }
    return 0;
}

ConnectionManager *ConnectionManagerFactory::getConnectionManagerByName(const QString &name) const
{
    ConnectionManager *cm;
    foreach (cm, m_cms) {
        if (cm->name() == name) {
            return cm;
        }
    }
    return 0;
}

void ConnectionManagerFactory::load()
{
    QStringList configDirs;
    char *dataPath = getenv("TELEPATHY_DATA_PATH");
    if (dataPath) {
        QStringList paths = QString(dataPath).split(":");
        QString path;
        foreach (path, paths) {
            configDirs << path + "/managers";
        }
    }
    configDirs << QString(QDir::homePath() + "/.telepathy/managers");
    configDirs << DATADIR "/telepathy/managers";
    configDirs << "/usr/local/share/telepathy/managers";
    configDirs << "/usr/share/telepathy/managers";

    QString dir;
    foreach (dir, configDirs) {
        QDir configDir(dir);
        QString name;
        QString file;
        QStringList files = configDir.entryList(QStringList("*.manager"), QDir::Files);
        foreach (file, files) {
            QString name = file.left(file.indexOf('.'));

            if (!getConnectionManagerByName(name)) {
                ConnectionManager *cm = new ConnectionManager(configDir.path() + "/" + file, name, this);
                m_cms.append(cm);
            }
        }
    }
}

void ConnectionManagerFactory::registerTypes()
{
    static bool registered = false;
    if (!registered) {
        // TapiocaQt Stuff.
        qDBusRegisterMetaType<QtTapioca::PresenceState>();

        // Register telepathy stuff:
        org::freedesktop::Telepathy::registerTypes();

        // General
        qDBusRegisterMetaType<QVariantMap>();
        // From: Connection_Interface_Presence_Types
        qDBusRegisterMetaType<org::freedesktop::Telepathy::ContactPresences>();
        qDBusRegisterMetaType<org::freedesktop::Telepathy::LastActivityAndStatuses>();
        qDBusRegisterMetaType<org::freedesktop::Telepathy::MultipleStatusMap>();
        // From: Connection_Interface_Capabilities_Types
        qDBusRegisterMetaType<org::freedesktop::Telepathy::CapabilityPair>();
        qDBusRegisterMetaType<org::freedesktop::Telepathy::ContactCapability>();
        qDBusRegisterMetaType<org::freedesktop::Telepathy::CapabilityChange>();
        // From: Connection_Intergace_Aliasing_Types
        qDBusRegisterMetaType<org::freedesktop::Telepathy::AliasPair>();
        // From: Connection_Types
        qDBusRegisterMetaType<org::freedesktop::Telepathy::ChannelInfo>();
        // From: Channel_Type_Text_Types
        qDBusRegisterMetaType<org::freedesktop::Telepathy::PendingTextMessage>();
        // From: Channel_Type_Streamed_Media_Types
        qDBusRegisterMetaType<org::freedesktop::Telepathy::MediaStreamInfo>();
        registered = true;
    }
}

