/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org>
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#include "config.h"

#include "QtTapioca/Connection"
#include "QtTapioca/Contact"
#include "QtTapioca/ContactList"
#include "QtTapioca/Handle"
#include "QtTapioca/HandleFactory"
#include "QtTapioca/TextChannel"
//#include "QtTapioca/StreamChannel"

#include <QtCore/QDebug>
#include <QtCore/QDir>
#include <QtCore/QMutex>
#include <QtCore/QSettings>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusConnectionInterface>
#include <QtDBus/QDBusMetaType>
#include <QtTelepathy/Client/Connection>
#include <QtTelepathy/Client/Connection_Interface_Avatars>
#include <QtTelepathy/Client/Connection_Interface_Presence>
#include <QtTelepathy/Client/Connection_Interface_Aliasing>
#include <QtTelepathy/Client/Connection_Interface_Capabilities>

namespace QtTapioca {

/*
 * Private class
 */
class ConnectionPrivate {
public:
    ConnectionPrivate(org::freedesktop::Telepathy::Connection * const telepathyConn,
                      QtTapioca::Connection * const p)
        : bus(QDBusConnection::sessionBus()),
          parent(p),
          conn(telepathyConn),
          iAvatar(0),
          iPresence(0),
          iAliasing(0),
          iCapabilities(0),
          cl(0),
          initialPresence(QtTapioca::PresenceState("Offline", QtTapioca::PresenceState::OfflineType, false)),
          uContact(0),
          selfHandle(0),
          handleFactory(new HandleFactory(telepathyConn, p))
    {
        Q_ASSERT(0 != telepathyConn);
        Q_ASSERT(0 != p);
        status = telepathyConn->GetStatus();
    }
    ~ConnectionPrivate()
    {
        delete iAvatar;
        delete iPresence;
        delete iAliasing;
        delete iCapabilities;
        delete uContact;
        delete cl;
        delete conn;
    }

    void loadInterfaces()
    {
        QStringList interfaces = conn->GetInterfaces();

        if (interfaces.contains(org::freedesktop::Telepathy::ConnectionInterfaceAvatars::staticInterfaceName()))
            iAvatar = new org::freedesktop::Telepathy::ConnectionInterfaceAvatars(conn->service(), conn->path(), bus);

        if (interfaces.contains(org::freedesktop::Telepathy::ConnectionInterfacePresence::staticInterfaceName()))
            iPresence = new org::freedesktop::Telepathy::ConnectionInterfacePresence(conn->service(), conn->path(), bus);

        if (interfaces.contains(org::freedesktop::Telepathy::ConnectionInterfaceAliasing::staticInterfaceName()))
            iAliasing = new org::freedesktop::Telepathy::ConnectionInterfaceAliasing(conn->service(), conn->path(), bus);

        if (interfaces.contains(org::freedesktop::Telepathy::ConnectionInterfaceCapabilities::staticInterfaceName()))
            iCapabilities = new org::freedesktop::Telepathy::ConnectionInterfaceCapabilities(conn->service(), conn->path(), bus);
    }

    void getSupportedPresenceStates()
    {
        // This method is called when the connection status changes to
        // Connected. We can now ask the CM what PresenceStates it supports
        // and cache them in this Connection object.

        // First we must check that the connection supports the presence interface.
        if(!iPresence)
        {
            return;
        }

        // Get the supported presences.
        org::freedesktop::Telepathy::StatusSpecMap statusSpecMap = iPresence->GetStatuses();

        // Iterate over all the supported presences that we got.
        org::freedesktop::Telepathy::StatusSpecMap::const_iterator end = statusSpecMap.constEnd();
        org::freedesktop::Telepathy::StatusSpecMap::const_iterator i;
        for(i = statusSpecMap.constBegin(); i != end; i++)
        {
            // For each presence state in the StatusSpecMap, create a
            // QtTapioca::PresenceState and insert it to the list of
            // supported PresenceStates.
            QtTapioca::PresenceState::Type presenceType = static_cast<QtTapioca::PresenceState::Type>(i.value().type);
            QtTapioca::PresenceState presenceState(i.key(), presenceType, i.value().maySetOnSelf);
            supportedPresenceStates.append(presenceState);
        }
    }

    void clearSupportedPresenceStates()
    {
        // This method is called when the connection changes status to
        // Disconnected. We remove all presence states from the list of
        // supported ones.
        supportedPresenceStates.clear();
    }

    QDBusConnection bus;
    QtTapioca::Connection * const parent;
    org::freedesktop::Telepathy::Connection * const conn;
    org::freedesktop::Telepathy::ConnectionInterfaceAvatars *iAvatar;
    org::freedesktop::Telepathy::ConnectionInterfacePresence *iPresence;
    org::freedesktop::Telepathy::ConnectionInterfaceAliasing *iAliasing;
    org::freedesktop::Telepathy::ConnectionInterfaceCapabilities *iCapabilities;
    uint status;
    ContactList *cl;
    QtTapioca::PresenceState initialPresence;
    QVariantMap initialParams;
    UserContact *uContact;
    Handle *selfHandle;
    HandleFactory * const handleFactory;
    QHash<QString, Channel *> channels;
    QMutex mutex;
    QList<QtTapioca::PresenceState> supportedPresenceStates;
};

}


using namespace QtTapioca;

/*
 * Constructor
 */
Connection::Connection(const QString &serviceName, const QString &objPath, QObject *parent)
    : DBusProxyObject(serviceName, objPath, parent),
      d(new ConnectionPrivate(new org::freedesktop::Telepathy::Connection(serviceName,
                                                                          objPath,
                                                                          QDBusConnection::sessionBus()),
                              this))
{
    Q_ASSERT(d);

    QObject::connect(d->conn, SIGNAL(NewChannel(const QDBusObjectPath &, const QString &, uint, uint, bool)),
                     this, SLOT(onNewChannel(const QDBusObjectPath &, const QString &, uint, uint, bool)));
    QObject::connect(d->conn, SIGNAL(StatusChanged(uint,uint)), this, SLOT(onStatusChanged(uint,uint)));

    updateOpenChannels();

    if (d->status == Connected) { d->loadInterfaces(); }
}

/*
 * Destructor
 */
Connection::~Connection()
{
    delete d;
}

/*
 * Get Protocol
 */
QString Connection::protocol() const
{
    return d->conn->GetProtocol();
}

/*
 * Get Status
 */
Connection::Status Connection::status() const
{
    return static_cast<Connection::Status>(d->status);
}

/*
 * Connect
 */
void Connection::connect(const QtTapioca::PresenceState & initialPresence, const QVariantMap &initialParams)
{
    if (d->status != Connection::Disconnected) { return; }

    d->initialPresence = initialPresence;
    d->initialParams = initialParams;
    d->conn->Connect();
}

/*
 * Disconnect
 */
void Connection::disconnect()
{
    if (d->status == Connection::Disconnected)
        return;
    d->conn->Disconnect();
}

/*
 * Create Channel
 */
Channel *Connection::createChannel(Channel::Type type, Contact *contact, bool suppress_handler)
{
    Channel *channel = 0;
    QString objPath;

    QMutexLocker lock(&(d->mutex));
    if (type == Channel::Text) {
        objPath = requestChannel("org.freedesktop.Telepathy.Channel.Type.Text", contact->handle(),
             suppress_handler);
        if (!objPath.isEmpty())
            channel = new TextChannel(this, serviceName(), objPath, contact, this);
    }
  /*  else if (type == Channel::Stream) {
        objPath = requestChannel("org.freedesktop.Telepathy.Channel.Type.StreamedMedia", contact->handle(),
             suppress_handler);
        if (!objPath.isEmpty())
            channel = new StreamChannel(this, serviceName(), objPath, contact, this);
    }*/

    if (channel) {
        d->channels[objPath] = channel;
        QObject::connect(channel, SIGNAL(destroyed()),
                         this, SLOT(onChannelDestroyed()));
    }

    return channel;
}

/*
 * Get Channels
 */
QList<Channel *> Connection::openChannels() const
{
    return d->channels.values();
}

/*
 * Request Channel
 */
QString Connection::requestChannel(const QString &interface, Handle *handle, bool suppress_handler)
{
    QDBusReply<QDBusObjectPath> channel = d->conn->RequestChannel(
            interface, handle->type(), handle->id(), suppress_handler);

    if (channel.isValid())
        return static_cast<QDBusObjectPath>(channel).path();

    return QString();
}

/*
 * Get Contact List
 */
ContactList *Connection::contactList()
{
    if (d->status == Connection::Disconnected)
    { return 0; }

    if (!d->cl) {
        d->cl = new ContactList(this,
                                d->conn,
                                d->iAvatar,
                                d->iPresence,
                                d->iAliasing,
                                d->iCapabilities,
                                d->handleFactory,
                                this);
    }
    return d->cl;
}

/*
 * Initialize UserContact.
 */
void Connection::initUserContact()
{
    if (!d->uContact) {
        if (!d->selfHandle) {
            /* Updated selfHandle */
            uint my_handle = d->conn->GetSelfHandle();
            d->selfHandle = d->handleFactory->createHandle(Handle::Contact, my_handle);
            Q_ASSERT (d->selfHandle != 0);
        }

        d->uContact = new UserContact(this, d->conn, d->iAvatar, d->iPresence, d->iAliasing, d->iCapabilities, d->selfHandle, this);

        d->uContact->setPresenceWithParameters(d->initialPresence, d->initialParams);
    }
}

/*
 * Get User Contact
 */
UserContact *Connection::userContact()
{
    if (d->uContact == 0 && d->status == Connected) { initUserContact(); }
    return d->uContact;
}

/*
 * has * Support
 */
bool Connection::hasAvatarSupport() const
{ return (0 != d->iAvatar); }

bool Connection::hasPresenceSupport() const
{ return (0 != d->iPresence); }

bool Connection::hasAliasingSupport() const
{ return (0 != d->iAliasing); }

bool Connection::hasCapabilitiesSupport() const
{ return (0 != d->iCapabilities); }

void Connection::updateOpenChannels()
{
    if (d->status != Connection::Connected)
    { return; }

    org::freedesktop::Telepathy::ChannelInfo channelInfo;
    QList<org::freedesktop::Telepathy::ChannelInfo> channelInfoList;

    channelInfoList = d->conn->ListChannels();

    foreach (channelInfo, channelInfoList) {
        incomingChannel(channelInfo.channel,
                        channelInfo.channelType,
                        channelInfo.handleType,
                        channelInfo.handle,
                        true);
    }
}

Channel * Connection::incomingChannel(const QDBusObjectPath &objPath,
                                      const QString &channelType,
                                      uint handleType,
                                      uint handleId,
                                      bool suppressHandler)
{
    Channel *channel = 0;

    QMutexLocker lock(&(d->mutex));

    if ((d->channels.contains(objPath.path())) ||
        ((channelType != "org.freedesktop.Telepathy.Channel.Type.Text") /*&&
        (channelType != "org.freedesktop.Telepathy.Channel.Type.StreamedMedia")*/))
    { return NULL; }

    Contact *contact = contactList()->contact(handleId);

    if (!contact) {
        Handle *handle = d->handleFactory->createHandle(handleType, handleId);
        contact = d->cl->addContact(handle);
        if (!contact) {
            qDebug() << "error creating a contact.";
            return NULL;
        }
    }

    if (channelType == "org.freedesktop.Telepathy.Channel.Type.Text")
        channel = new TextChannel(this, serviceName(), objPath.path(), contact, this);
  //  else if (channelType == "org.freedesktop.Telepathy.Channel.Type.StreamedMedia")
    //    channel = new StreamChannel(this, serviceName(), objPath.path(), contact, this);

    if (channel) {
        d->channels[objPath.path()] = channel;
        QObject::connect(channel, SIGNAL(destroyed()),
                         this, SLOT(onChannelDestroyed()));
    }

    return channel;
}

/*
 * On New Channel
 */
void Connection::onNewChannel(const QDBusObjectPath &objPath,
                              const QString &channelType,
                              uint handleType,
                              uint handleId,
                              bool suppressHandler)
{
    Channel *channel = incomingChannel(objPath,
                                       channelType,
                                       handleType,
                                       handleId,
                                       suppressHandler);

    if (channel)
        emit channelCreated(this, channel, suppressHandler);
}

/*
 * On Status Changed
 */
void Connection::onStatusChanged(uint status, uint reason)
{
    d->status = status;

    if (status == Connected) {
        d->loadInterfaces();
        d->getSupportedPresenceStates();
    }

    emit statusChanged(this,
                       static_cast<Connection::Status>(status),
                       static_cast<Connection::Reason>(reason));
    if (status == Disconnected) {
        /* we are no longer valid */
     //   deleteLater();    // HACK: Stops a SEGFAULT, in decibel, but probably causes a memory leak.
        d->clearSupportedPresenceStates();
    }
}

/*
 * Channel Destroyed
 */
void Connection::onChannelDestroyed()
{
    Channel *channel = static_cast<Channel *>(sender());
    d->channels.remove(channel->objectPath());
}

QList<QtTapioca::PresenceState> Connection::supportedPresenceStates() const
{
    return d->supportedPresenceStates;
}

QtTapioca::PresenceState Connection::presenceStateFromName(const QString & name) const
{
    // Loop through all the supported presence states, looking for one that's
    // name matches the name requested. If we find one, return it. Otherwise,
    // return an invalid one.
    foreach(const QtTapioca::PresenceState state, d->supportedPresenceStates)
    {
        if(state.name() == name)
        {
            return state;
        }
    }
    return QtTapioca::PresenceState(QString());
}

