/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#ifndef QTTAPIOCA_CONTACT_LIST_H
#define QTTAPIOCA_CONTACT_LIST_H

#include <QtTapioca/TapiocaExport>

#include <QtCore/QList>
#include <QtCore/QObject>
#include <QtCore/QString>

#include <QtTelepathy/Client/Connection>
#include <QtTelepathy/Client/Connection_Interface_Avatars>
#include <QtTelepathy/Client/Connection_Interface_Presence>
#include <QtTelepathy/Client/Connection_Interface_Aliasing>
#include <QtTelepathy/Client/Connection_Interface_Capabilities>

/** Namespace for Tapioca classes. */
namespace QtTapioca {

class ContactListPrivate;
class Contact;
class Handle;
class HandleFactory;
class Connection;

/**
 * This class is holds all known contacts. New contacts are also created here.
 */
class QTTAPIOCA_EXPORT ContactList: public QObject {
    Q_OBJECT
    Q_DECLARE_PRIVATE(ContactList)

public:
    /** Destructor */
    ~ContactList();

    /**
     * Adds a new contact to the list.
     *
     * @param uri Contact's uri. e.g. "foo@jabber.org"
     * @return The specified contact.
     */
    Contact *addContact(const QString &uri);

    /**
     * Removes a contact from the list.
     *
     * @param contact The contact to be removed.
     */
    void removeContact(Contact *contact);

    Contact *contact(const QString &uri) const;

    QList<Contact *> knownContacts() const;
    QList<Contact *> subscribedContacts() const;
    QList<Contact *> authorizedContacts() const;
    QList<Contact *> blockedContacts() const;
    QList<Contact *> hiddenContacts() const;

    /**
     * Whether it's stored on server (FIXME)
     * @return true or false
     */
    bool isStoredOnServer() const;

Q_SIGNALS:
    void authorizationRequested(QtTapioca::Contact *contact);
    void subscriptionAccepted(QtTapioca::Contact *contact);

private Q_SLOTS:
    void onMembersChangedSubscribe(const QString &, const QList<quint32> &, const QList<quint32> &, const QList<quint32> &, const QList<quint32> &, uint, uint);
    void onMembersChangedPublish(const QString &, const QList<quint32> &, const QList<quint32> &, const QList<quint32> &, const QList<quint32> &, uint, uint);

private:
    friend class Connection;
    friend class ContactGroup;

    ContactList(Connection *connection,
                org::freedesktop::Telepathy::Connection *telepathyConn,
                org::freedesktop::Telepathy::ConnectionInterfaceAvatars *iAvatar,
                org::freedesktop::Telepathy::ConnectionInterfacePresence *iPresence,
                org::freedesktop::Telepathy::ConnectionInterfaceAliasing *iAliasing,
                org::freedesktop::Telepathy::ConnectionInterfaceCapabilities *iCapabilities,
                HandleFactory *handleFactory,
                QObject *parent = 0);

    void loadContacts();
    Contact *contact(Handle *handle) const;
    Contact *contact(uint id) const;
    Contact *addContact(const Handle *handle);
    Contact *addContact(const uint id);
    Connection *m_connection;

    ContactListPrivate *d;
};

} // namespace

#endif

