/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.h>
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#ifndef QTTAPIOCA_CONNECTION_H
#define QTTAPIOCA_CONNECTION_H

#include <QtTapioca/TapiocaExport>

#include <QtCore/QString>
#include <QtDBus/QDBusObjectPath>

#include <QtTapioca/DBusProxyObject>
#include <QtTapioca/Channel>
#include <QtTapioca/ContactBase>
#include <QtTapioca/UserContact>
#include <QtTapioca/PresenceState>

/** Namespace for Tapioca classes. */
namespace QtTapioca {

class ConnectionPrivate;
class Contact;
class ContactList;
class Handle;

/**
 * A Tapioca Connection. It holds the contact list and the handle factory. All channels are created here.
 */
class QTTAPIOCA_EXPORT Connection: public DBusProxyObject {
    Q_OBJECT
    Q_PROPERTY(QString protocol READ protocol)
    Q_DECLARE_PRIVATE(Connection)
    Q_ENUMS(Status Reason)

public:
    enum Status {
        Connected = 0,
        Connecting,
        Disconnected
    };

    enum Reason {
        NoneSpecified = 0,
        Requested,
        NetworkError,
        AuthenticationFailed,
        EncryptionError,
        NameInUse,
        CertificateNotProvided,
        CertificateUntrusted,
        CertificateExpired,
        CertificateNotActivated,
        CertificateHostnameMismatch,
        CertificateFingerPrintMismatch,
        CertificateSigned,
        CertificateOtherError
    };

    /** Constructor */
    Connection(const QString &serviceName, const QString &objPath, QObject *parent = 0);
    /** Destructor */
    ~Connection();

    /** Get connection protocol. */
    QString protocol() const;

    /** Get connection status. */
    Status status() const;

    /** Connect with a specified initial presence.
     *
     * @param initialPresence Connection's initial presence.
     */
    void connect(const QtTapioca::PresenceState & initialPresence,
                 const QVariantMap &initialParams = QVariantMap());

    /**
     * Disconnects.
     */
    void disconnect();

    Channel *createChannel(Channel::Type type, Contact *contact, bool suppress_handler = true);
    QList<Channel *> openChannels() const;

    /** Gets the contact list.
     *
     * @return Connection's contact list.
     */
    ContactList *contactList();

    /** Get user's own contact.
     *
     *  @return User's own contact.
     */
    UserContact *userContact();

    /** Check for Avatar interface support.
     *
     *  @return true is avatar interface is present and false otherwise.
     */
    bool hasAvatarSupport() const;

    /** Check for presence interface support.
     *
     *  @return true is presence interface is present and false otherwise.
     */
    bool hasPresenceSupport() const;

    /** Check for Aliasing interface support.
     *
     *  @return true is aliasing interface is present and false otherwise.
     */
    bool hasAliasingSupport() const;

    /** Check for Capabilities interface support.
     *
     *  @return true is capabilities interface is present and false otherwise.
     */
    bool hasCapabilitiesSupport() const;

    /**
     */
    QList<QtTapioca::PresenceState> supportedPresenceStates() const;

    /**
     */
    QtTapioca::PresenceState presenceStateFromName(const QString & name) const;

Q_SIGNALS:
    void statusChanged(QtTapioca::Connection *conn, QtTapioca::Connection::Status status, QtTapioca::Connection::Reason reason);
    void channelCreated(QtTapioca::Connection *conn, QtTapioca::Channel *channel, bool suppress_handler);

private Q_SLOTS:
    void onNewChannel(const QDBusObjectPath &, const QString &, uint, uint, bool);
    void onStatusChanged(uint, uint);
    void onChannelDestroyed();

private:
    QString requestChannel(const QString &interface, Handle *handle, bool suppress_handler);
    void initUserContact();
    void updateOpenChannels();
    Channel *incomingChannel(const QDBusObjectPath &, const QString &, uint, uint, bool);

    ConnectionPrivate * const d;
};

} // namespace

#endif

