// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include <math.h>

#include "main.h"

template< typename T, int Size, typename MatrixType, typename VectorType >
void helper_checkScaling( const char *strType )
{
    int i;
    T a;
    pickRandom(a);
    MatrixType m( Size );
    VectorType u( Size), v( Size ), w( Size );
    v.loadRandom();
    m.loadScaling( a );
    QVERIFY( compareVectors( a * v, m * v, strType ) );
    m.loadDiagonal( a );
    QVERIFY( compareVectors( a * v, m * v, strType ) );
    w.loadRandom();
    m.loadScaling( w );
    u = m * v;
    for( i = 0; i < Size; i++ )
        QVERIFY( test_isApprox( u(i), w(i) * v(i) ) );
    m.loadDiagonal( w );
    u = m * v;
    for( i = 0; i < Size; i++ )
        QVERIFY( test_isApprox( u(i), w(i) * v(i) ) );
    MatrixType m2( Size ), m3( Size );
    m.loadIdentity();
    if( Size > 1 ) m(1, 0) = static_cast<T>(1);
    m2 = m;
    m2.scale(v);
    m3.loadScaling(v);
    m3 = m * m3;
    QVERIFY( compareMatrices( m2, m3, strType ) );
    m2 = m;
    m2.prescale(v);
    m3.loadScaling(v);
    m3 *= m;
    QVERIFY( compareMatrices( m2, m3, strType ) );
}

template< typename T, typename MatrixType, typename VectorType >
void helper_checkRotation2( const char *strType )
{
    T a;
    pickRandom(a);
    MatrixType m(2), id(2);
    m.loadRotation2(a);
    id.loadIdentity();
    QVERIFY( compareMatrices( m * m.adjoint(), id, strType ) );
    VectorType v(2);
    v.loadRandomUnit();
    QVERIFY( test_isApprox( dot( v, m*v ), cos(a) ) );
    v(0) = static_cast<T>(1);
    v(1) = static_cast<T>(0);
    a = static_cast<T>( M_PI / 2 );
    m.loadRotation2(a);
    v = m * v;
    QVERIFY( test_isApprox( v(1), static_cast<T>(1) ) );

    T a2;
    pickRandom(a2);
    MatrixType m2(2);
    m.loadRotation2(a);
    m.rotate2(a2);
    m2.loadRotation2(a2);
    m2.prerotate2(a);
    QVERIFY( compareMatrices( m, m2, strType ) );
}

template< typename T, typename MatrixType, typename VectorType >
void helper_checkRotation3( const char *strType )
{
    T a;
    pickRandom(a);
    MatrixType m(3), id(3);
    VectorType u(3), v(3), w(3);
    u.loadRandomUnit();
    m.loadRotation3( a, u );
    id.loadIdentity();
    QVERIFY( compareMatrices( m * m.adjoint(), id, strType ) );
    v.loadRandom();
    QVERIFY( test_isApprox( dot( u, v ), dot( u, m*v ) ) );
    u(0) = static_cast<T>(1);
    u(1) = static_cast<T>(0);
    u(2) = static_cast<T>(0);
    v(0) = static_cast<T>(0);
    v(1) = static_cast<T>(1);
    v(2) = static_cast<T>(0);
    a = M_PI / 2;
    m.loadRotation3(a, u);
    w = m * v;
    QVERIFY( test_isApprox( w(2), static_cast<T>(1) ) );

    T a2;
    pickRandom(a2);
    VectorType u2(3);
    u.loadRandomUnit();
    u2.loadRandomUnit();
    MatrixType m2(3);
    m.loadRotation3(a, u);
    m.rotate3(a2, u2);
    m2.loadRotation3(a2, u2);
    m2.prerotate3(a, u);
    QVERIFY( compareMatrices( m, m2, strType ) );
}

void checkScaling()
{
#define HELPER_CHECKSCALING_FIX( T, Size ) \
    helper_checkScaling< T, Size, \
        Matrix< T, Size >, Vector< T, Size > >\
            STRINGY("fixed",T,Size)

#define HELPER_CHECKSCALING_DYN( T, Size ) \
    helper_checkScaling< T, Size, \
        MatrixX< T >, VectorX< T > >\
            STRINGY("dynamic",T,Size)

    HELPER_CHECKSCALING_FIX( double, 2 );
    HELPER_CHECKSCALING_FIX( float, 3 );
    HELPER_CHECKSCALING_FIX( double, 4 );
    HELPER_CHECKSCALING_FIX( double, 7 );
    HELPER_CHECKSCALING_FIX( float, 8 );
    HELPER_CHECKSCALING_FIX( double, 12 );
/*
    HELPER_CHECKSCALING_FIX( complex<double>, 7 );
    HELPER_CHECKSCALING_FIX( complex<float>, 8 );
    HELPER_CHECKSCALING_FIX( complex<double>, 12 );

    HELPER_CHECKSCALING_DYN( double, 2 );
    HELPER_CHECKSCALING_DYN( float, 3 );
    HELPER_CHECKSCALING_DYN( double, 4 );
    HELPER_CHECKSCALING_DYN( double, 7 );
    HELPER_CHECKSCALING_DYN( float, 8 );
    HELPER_CHECKSCALING_DYN( double, 12 );
    HELPER_CHECKSCALING_DYN( complex<double>, 7 );
    HELPER_CHECKSCALING_DYN( complex<float>, 8 );
    HELPER_CHECKSCALING_DYN( complex<double>, 12 );
*/}

void checkRotation2()
{
#define HELPER_CHECKROTATION2_FIX( T ) \
    helper_checkRotation2< T, \
        Matrix<T,2>, Vector<T,2> >\
            STRINGY("fixed",T,2)

#define HELPER_CHECKROTATION2_DYN( T ) \
    helper_checkRotation2< T, \
        MatrixX< T >, VectorX< T > >\
            STRINGY("dynamic",T,2)

    HELPER_CHECKROTATION2_FIX( double );
    HELPER_CHECKROTATION2_FIX( float );
    HELPER_CHECKROTATION2_DYN( double );
    HELPER_CHECKROTATION2_DYN( float );
}

void checkRotation3()
{
#define HELPER_CHECKROTATION3_FIX( T ) \
    helper_checkRotation3< T, \
        Matrix<T,3>, Vector<T,3> >\
            STRINGY("fixed",T,3)

#define HELPER_CHECKROTATION3_DYN( T ) \
    helper_checkRotation3< T, \
        MatrixX<T>, VectorX<T> >\
            STRINGY("dynamic",T,3)

    HELPER_CHECKROTATION3_FIX( double );
    HELPER_CHECKROTATION3_FIX( float );
    HELPER_CHECKROTATION3_DYN( double );
    HELPER_CHECKROTATION3_DYN( float );
}

void MainTest::checkSpecialMatrices()
{
    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        checkScaling();
        checkRotation2();
        checkRotation3();
    }
}
