// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include "main.h"

template< typename T, int Size, typename MatrixType, typename VectorType >
void helper_checkRowColumnAdjoint( const char *strType )
{
    VectorType vectors[Size], v;
    MatrixType mat1(Size), mat2(Size), mat3(Size);
    int i, j;
    for( i = 0; i < Size; i++ )
    {
        vectors[i].resize(Size);
        pickRandomVector( vectors[i] );
        mat1.setRow( i, vectors[i] );
        mat2.setColumn( i, vectors[i] );

        QVERIFY( compareVectors( vectors[i], mat1.row(i), strType ) );
        mat1.getRow( i, & v );
        QVERIFY( compareVectors( vectors[i], v, strType ) );

        QVERIFY( compareVectors( vectors[i], mat2.column(i), strType ) );
        mat2.getColumn( i, & v );
        QVERIFY( compareVectors( vectors[i], v, strType ) );
    }

    for( i = 0; i < Size; i++ )
        for( j = 0; j < Size; j++ )
            mat2( i, j ) = Util::conj( mat2( i, j ) );

    mat1.computeAdjoint( & mat3 );
    QVERIFY( compareMatrices( mat2, mat3, strType ) );
    QVERIFY( compareMatrices( mat1.adjoint(), mat3, strType ) );
    mat1.replaceWithAdjoint();
    QVERIFY( compareMatrices( mat1, mat2, strType ) );
}

void MainTest::checkRowColumnAdjoint()
{
#define HELPER_CHECKROWCOLUMNADJOINT_FIX( T, Size ) \
    helper_checkRowColumnAdjoint< T, Size, \
        Matrix< T, Size >, Vector< T, Size > >\
            STRINGY("fixed",T,Size)

#define HELPER_CHECKROWCOLUMNADJOINT_DYN( T, Size ) \
    helper_checkRowColumnAdjoint< T, Size, \
        MatrixX< T >, VectorX< T > >\
            STRINGY("dynamic",T,Size)

    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        HELPER_CHECKROWCOLUMNADJOINT_FIX( float, 1 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( double, 2 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( float, 3 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( double, 4 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( double, 7 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( float, 8 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( double, 12 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( complex<double>, 7 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( complex<float>, 8 );
        HELPER_CHECKROWCOLUMNADJOINT_FIX( complex<double>, 12 );
    
        HELPER_CHECKROWCOLUMNADJOINT_DYN( float, 1 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( double, 2 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( float, 3 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( double, 4 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( double, 7 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( float, 8 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( double, 12 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( complex<double>, 7 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( complex<float>, 8 );
        HELPER_CHECKROWCOLUMNADJOINT_DYN( complex<double>, 12 );
    }
}
