// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include "main.h"
#include "../src/projective.h"

template< typename T, int Size >
bool compareMatrixMatrixP( const Matrix<T, Size> & m,
                           const MatrixP<T, Size> & mp,
                           const char *strType )
{
    int i, j;

    for( i = 0; i < Size; i++ )
    {
        for( j = 0; j < Size; j++ )
            if( ! test_isApprox_relax( mp( i, j), m( i, j ) ) )
                goto _false;
        if( ! test_isApprox_relax( mp( Size, i ), static_cast<T>(0) ) )
            goto _false;
        if( ! test_isApprox_relax( mp( i, Size ), static_cast<T>(0) ) )
            goto _false;
    }
    if( ! test_isApprox_relax( mp( Size, Size ), static_cast<T>(1) ) )
        goto _false;
    return true;
_false:
    qDebug() <<"Type =" << strType;
    return false;
}

template< typename T, int Size >
void helper_checkProjective( const char *strType )
{
    int i;
    Matrix<T, Size> msmall, msmall2;
    msmall.loadRandom();
    MatrixP<T, Size> mp1( msmall );
    QVERIFY( compareMatrixMatrixP( msmall, mp1, strType ) );
    Matrix<T, Size+1> mbig;
    mbig.loadRandom();
    MatrixP<T, Size> mp2( mbig );
    QVERIFY( compareMatrices( mp2.matrix(), mbig, strType ) );
    MatrixP<T, Size> mp3( mp2 );
    QVERIFY( compareMatrices( mp2.matrix(), mp3.matrix(), strType ) );
    Vector<T, Size> v, v2;
    v.loadRandom();
    msmall.loadScaling(v);
    mp2.loadScaling(v);
    QVERIFY( compareMatrixMatrixP( msmall, mp2, strType ) );
    mp3.loadTranslation(v);
    for( i = 0; i < Size; i++ ) mp3( i, Size ) -= v(i);
    mbig.loadIdentity();
    QVERIFY( compareMatrices( mp3.matrix(), mbig, strType ) );

    v.loadRandom();
    mp3.setTranslationVector(v);
    mp3.getTranslationVector(&v2);
    QVERIFY( compareVectors( v, v2, strType ) );
    QVERIFY( compareVectors( v, mp3.translationVector(), strType ) );
    mp3.loadTranslation(v);
    v2.loadZero();
    mp3.linearMultiply( v, &v2 );

    msmall.loadRandom();
    mp3.loadMatrix(msmall);
    mp3.getLinearComponent(&msmall2);
    QVERIFY( compareMatrices( msmall, msmall2, strType ) );
    mp3.setLinearComponent(msmall2);
    QVERIFY( compareMatrices( msmall, mp3.linearComponent(), strType ) );

    v.loadRandom();
    mp3.loadTranslation(v);
    Vector<T, Size> vs, vs2;
    Vector<T, Size+1> vb;
    vs.loadRandom();
    QVERIFY( compareVectors( mp3 * vs, vs + v, strType ) );
    affToProj( vs, & vb );
    vb = mp3 * vb;
    projToAff( vb, & vs2 );
    QVERIFY( compareVectors( vs2, vs + v, strType ) );

    mp2.loadScaling(v);
    mp3.loadScaling(static_cast<T>(1)).scale(v);
    QVERIFY( compareMatrices( mp2.matrix(), mp3.matrix(), strType ) );

    mp2.loadTranslation(v);
    mp3.loadIdentity().translate(v);
    QVERIFY( compareMatrices( mp2.matrix(), mp3.matrix(), strType ) );

    mp1 = mp2 * mp3;
    mp2 *= mp3;
    QVERIFY( compareMatrices( mp2.matrix(), mp1.matrix(), strType ) );

    vs.loadRandom();
    vs2.loadRandom();
    mp1.loadTranslation(vs);
    mp1.scale(vs2);
    mp2.loadScaling(vs2);
    mp2.pretranslate(vs);
    QVERIFY( compareMatrices( mp2.matrix(), mp1.matrix(), strType ) );
}

template< typename T >
void helper_checkProjectiveRotation3( const char *strType )
{
    T a;
    pickRandom(a);
    Vector<T, 3> v;
    v.loadRandomUnit();
    Matrix<T, 3> m;
    MatrixP<T, 3> mp, mp2;
    m.loadRotation3( a, v );
    mp.loadRotation3( a, v );
    mp2.loadIdentity().rotate3(a, v);
    QVERIFY( compareMatrixMatrixP( m, mp, strType ) );
    QVERIFY( compareMatrixMatrixP( m, mp2, strType ) );

    T a2;
    pickRandom(a2);
    Vector<T, 3> v2;
    v2.loadRandomUnit();
    mp.loadRotation3( a, v );
    mp.prerotate3( a2, v2 );
    mp2.loadRotation3( a2, v2 );
    mp2.rotate3( a, v );
    QVERIFY( compareMatrices( mp2.matrix(), mp.matrix(), strType ) );
}

template< typename T >
void helper_checkProjectiveRotation2( const char *strType )
{
    T a;
    pickRandom(a);
    Matrix<T, 2> m;
    MatrixP<T, 2> mp, mp2, mp3;
    m.loadRotation2( a );
    mp.loadRotation2( a );
    mp2.loadIdentity().rotate2(a);
    mp3.loadIdentity().prerotate2(a);
    QVERIFY( compareMatrixMatrixP( m, mp, strType ) );
    QVERIFY( compareMatrixMatrixP( m, mp2, strType ) );
    QVERIFY( compareMatrixMatrixP( m, mp3, strType ) );
}

void MainTest::checkProjective()
{
#define HELPER_CHECKPROJECTIVE( T, Size ) \
    helper_checkProjective< T, Size >\
            STRINGY("fixed",T,Size)

#define HELPER_CHECKPROJECTIVEROTATION3( T ) \
    helper_checkProjectiveRotation3< T >\
            STRINGY("fixed",T,3)

#define HELPER_CHECKPROJECTIVEROTATION2( T ) \
    helper_checkProjectiveRotation2< T >\
            STRINGY("fixed",T,2)

    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        HELPER_CHECKPROJECTIVE( double, 2 );
        HELPER_CHECKPROJECTIVE( float, 3 );
        HELPER_CHECKPROJECTIVE( double, 4 );
        HELPER_CHECKPROJECTIVE( double, 7 );
        HELPER_CHECKPROJECTIVE( float, 8 );
        HELPER_CHECKPROJECTIVE( double, 12 );
        HELPER_CHECKPROJECTIVE( complex<double>, 7 );
        HELPER_CHECKPROJECTIVE( complex<float>, 8 );
        HELPER_CHECKPROJECTIVE( complex<double>, 12 );
        HELPER_CHECKPROJECTIVEROTATION3( double );
        HELPER_CHECKPROJECTIVEROTATION3( float );
        HELPER_CHECKPROJECTIVEROTATION2( double );
        HELPER_CHECKPROJECTIVEROTATION2( float );
    }
}
