// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#ifndef EIGEN_TESTS_MAIN_H
#define EIGEN_TESTS_MAIN_H

#include <QtTest/QtTest>
#include <iostream>
#include <complex>
#include <stdlib.h>
#include <time.h>

#include "../src/matrix.h"

#define REPEAT 30

#ifdef __GNUC__
#define STRINGY(label,T,Size) \
    (label"<"__STRING(T) ","__STRING(Size)">")
#else
#define STRINGY(label,T,Size) \
    ( label"<" #T "," #Size ">" )
#endif

using namespace Eigen;
using namespace std;

class MainTest : public QObject
{
    Q_OBJECT

public:
    MainTest();

private slots:
    void checkComparisons();
    void checkBasicStuff();
    void checkRowColumnAdjoint();
    void checkEuclidean();
    void checkLUDecomposition();
    void checkImplicitLU();
    void checkSolver();
    void checkRegression();
    void checkSpecialMatrices();
    void checkProjective();
};

// these tests really aren't about precision. If the result is remotely sane,
// it's good enough.
template<typename T> inline bool test_isNegligible( const T& a, const T& b )
{
    return std::abs(a) <= std::abs(b * static_cast<T>(1000) * Util::epsilon<T>() );
}

template<typename T> inline bool test_isApprox( const T& a, const T& b )
{
    return test_isNegligible( std::abs( a - b ),
                              std::min( std::abs(a), std::abs(b) ) );
}

template<typename T> inline bool test_isApprox_relax( const T& a, const T& b )
{
    return test_isApprox( a, b ) || test_isApprox( a + static_cast<T>(1),
                                                   b + static_cast<T>(1) );
}

template< typename T > void pickRandom( T & x )
{
    Util::pickRandom( x );
}

template<typename MatrixType> void pickRandomMatrix( MatrixType & m )
{
    for( int i = 0; i < m.size(); i++ )
        for( int j = 0; j < m.size(); j++ )
            pickRandom( m(i,j) );
}

template<typename VectorType> void pickRandomVector( VectorType & v )
{
    for( int i = 0; i < v.size(); i++ )
            pickRandom( v(i) );
}

template<typename VectorType> bool compareVectors
( const VectorType & v1, const VectorType & v2, const char *strType )
{
    if( ! v1.isApprox(v2, static_cast<typename VectorType::ScalType>(1000) * Util::epsilon<typename VectorType::ScalType>() ) ) goto _false;
    return true;
_false:
    qDebug() <<"Type =" << strType
             <<", relative error:" << std::abs( (v1 - v2).norm() / v1.norm() );
    return false;
}

template<typename MatrixType> bool compareMatrices
( const MatrixType & m1, const MatrixType & m2, const char *strType )
{
    if( ! m1.isApprox(m2, static_cast<typename MatrixType::ScalType>(1000) * Util::epsilon<typename MatrixType::ScalType>() ) ) goto _false;
    return true;
_false:
    qDebug() <<"Type =" << strType;
    return false;
}

#endif
