// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.

#include "main.h"

template< typename T, int Size, typename LUDecompositionType,
          typename MatrixType, typename VectorType >
void helper_checkLUDecomposition( const char *strType )
{
    MatrixType mat1(Size), mat2(Size), mat3(Size);
    VectorType vec1(Size), vec2(Size);
    do pickRandomMatrix( mat1 ); while( ! mat1.isInvertible() );
    mat2.loadIdentity();
    LUDecompositionType lu1(mat1);
    lu1.computeInverse( & mat3 );
    QVERIFY( compareMatrices( mat2, mat1 * mat3, strType ) );
    LUDecompositionType lu3(mat3);
    QVERIFY( Size == lu3.rank() );
    QVERIFY( 0 == lu3.dimKer() );
    do pickRandomMatrix( mat2 ); while( ! mat2.isInvertible() );
    LUDecompositionType lu2(mat2);
    LUDecompositionType lu4(mat1 * mat2);
    QVERIFY( test_isApprox_relax( lu4.determinant(), lu1.determinant() * lu2.determinant() ) );
}

void MainTest::checkLUDecomposition()
{
#define HELPER_CHECKLUDECOMPOSITION_FIX( T, Size ) \
    helper_checkLUDecomposition< T, Size, LUDecomposition<T, Size>,\
        Matrix< T, Size >, Vector< T, Size > >\
            STRINGY("fixed",T,Size)

#define HELPER_CHECKLUDECOMPOSITION_DYN( T, Size ) \
    helper_checkLUDecomposition< T, Size, LUDecompositionX< T >,\
        MatrixX< T >, VectorX< T > >\
            STRINGY("dynamic",T,Size)

    for( int repeat = 0; repeat < REPEAT; repeat++ )
    {
        HELPER_CHECKLUDECOMPOSITION_FIX( float, 1 );
        HELPER_CHECKLUDECOMPOSITION_FIX( double, 2 );
        HELPER_CHECKLUDECOMPOSITION_FIX( float, 3 );
        HELPER_CHECKLUDECOMPOSITION_FIX( double, 4 );
        HELPER_CHECKLUDECOMPOSITION_FIX( double, 7 );
        HELPER_CHECKLUDECOMPOSITION_FIX( float, 8 );
        HELPER_CHECKLUDECOMPOSITION_FIX( double, 12 );
        HELPER_CHECKLUDECOMPOSITION_FIX( complex<double>, 7 );
        HELPER_CHECKLUDECOMPOSITION_FIX( complex<float>, 8 );
        HELPER_CHECKLUDECOMPOSITION_FIX( complex<double>, 12 );
    
        HELPER_CHECKLUDECOMPOSITION_DYN( float, 1 );
        HELPER_CHECKLUDECOMPOSITION_DYN( double, 2 );
        HELPER_CHECKLUDECOMPOSITION_DYN( float, 3 );
        HELPER_CHECKLUDECOMPOSITION_DYN( double, 4 );
        HELPER_CHECKLUDECOMPOSITION_DYN( double, 7 );
        HELPER_CHECKLUDECOMPOSITION_DYN( float, 8 );
        HELPER_CHECKLUDECOMPOSITION_DYN( double, 12 );
        HELPER_CHECKLUDECOMPOSITION_DYN( complex<double>, 7 );
        HELPER_CHECKLUDECOMPOSITION_DYN( complex<float>, 8 );
        HELPER_CHECKLUDECOMPOSITION_DYN( complex<double>, 12 );
    }
}

