// This file is part of Eigen, a lightweight C++ template library
// for linear algebra. Eigen itself is part of the KDE project.
//
// Copyright (C) 2004-2007 Torsten Rahn <tackat@kde.org>
// Copyright (C) 2007 Casper Boemann <cbr@boemann.dk>
// Copyright (C) 2006-2007 Benoit Jacob <jacob@math.jussieu.fr>
//
// Eigen is free software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 or (at your option) any later version.
//
// Eigen is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
// details.
//
// You should have received a copy of the GNU General Public License along
// with Eigen; if not, write to the Free Software Foundation, Inc., 51
// Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. This exception does not invalidate any other reasons why a work
// based on this file might be covered by the GNU General Public License.
//

// Quaternions provides a class that deals with quaternion operations.

#ifndef EIGEN_QUATERNION_H
#define EIGEN_QUATERNION_H

namespace Eigen
{

template< typename T >
class Quaternion {
 public:
    /**
      * Default constructor. Constructs a quaternion with uninitialized coords.
      */
    Quaternion() {}

    /**
      * Convenience constructor
      */
    Quaternion(T w, T x, T y, T z)
    {
        m_array[Q_X] = x;
        m_array[Q_Y] = y;
        m_array[Q_Z] = z;
        m_array[Q_W] = w;
    }

    /**
      * Convenience constructor
      * Takes euler angles
      */
    Quaternion(float alpha, float beta)
    {
        m_array[Q_W] = 0.0f;

        const T  cosBeta = cosf(beta);
        m_array[Q_X] = -cosBeta * sinf(alpha);
        m_array[Q_Y] = -sinf(beta);
        m_array[Q_Z] = cosBeta * cosf(alpha);
    }

    /**
      * Copy constructor.
      */
    Quaternion( const Quaternion &other )
    {
        m_array[Q_X] = other.m_array[Q_X];
        m_array[Q_Y] = other.m_array[Q_Y];
        m_array[Q_Z] = other.m_array[Q_Z];
        m_array[Q_W] = other.m_array[Q_W];
    }


    virtual ~Quaternion(){ }

    Quaternion  operator * (const Quaternion &q) const
    {
        T  w, x, y, z;

        w = m_array[Q_W] * q.m_array[Q_W] - m_array[Q_X] * q.m_array[Q_X] - m_array[Q_Y] * q.m_array[Q_Y] - m_array[Q_Z] * q.m_array[Q_Z];
        x = m_array[Q_W] * q.m_array[Q_X] + m_array[Q_X] * q.m_array[Q_W] + m_array[Q_Y] * q.m_array[Q_Z] - m_array[Q_Z] * q.m_array[Q_Y];
        y = m_array[Q_W] * q.m_array[Q_Y] - m_array[Q_X] * q.m_array[Q_Z] + m_array[Q_Y] * q.m_array[Q_W] + m_array[Q_Z] * q.m_array[Q_X];
        z = m_array[Q_W] * q.m_array[Q_Z] + m_array[Q_X] * q.m_array[Q_Y] - m_array[Q_Y] * q.m_array[Q_X] + m_array[Q_Z] * q.m_array[Q_W];
        return Quaternion( w, x, y, z );
    }

    bool operator == (const Quaternion &q) const;
    {
        return ( m_array[Q_W] == q.m_array[Q_W]
            && m_array[Q_X] == q.m_array[Q_X]
            && m_array[Q_Y] == q.m_array[Q_Y]
            && m_array[Q_Z] == q.m_array[Q_Z] );
    }

    void operator *= (const Quaternion &q)
    {
        T x, y, z, w;

        w = m_array[Q_W] * q.m_array[Q_W] - m_array[Q_X] * q.m_array[Q_X] - m_array[Q_Y] * q.m_array[Q_Y] - m_array[Q_Z] * q.m_array[Q_Z];
        x = m_array[Q_W] * q.m_array[Q_X] + m_array[Q_X] * q.m_array[Q_W] + m_array[Q_Y] * q.m_array[Q_Z] - m_array[Q_Z] * q.m_array[Q_Y];
        y = m_array[Q_W] * q.m_array[Q_Y] - m_array[Q_X] * q.m_array[Q_Z] + m_array[Q_Y] * q.m_array[Q_W] + m_array[Q_Z] * q.m_array[Q_X];
        z = m_array[Q_W] * q.m_array[Q_Z] + m_array[Q_X] * q.m_array[Q_Y] - m_array[Q_Y] * q.m_array[Q_X] + m_array[Q_Z] * q.m_array[Q_W];

        m_array[Q_W] = w;
        m_array[Q_X] = x;
        m_array[Q_Y] = y;
        m_array[Q_Z] = z;
    }

    /**
      * Returns *this * factor (multiplication of each coord).
      *
      * \return_by_value
      *
      * \sa operator*=(const T&)
      */
    Quaternion operator * ( const T & factor ) const
    {
        return Quaternion( m_array[Q_W] *mult, m_array[Q_X] * mult,  m_array[Q_Y] * mult,  m_array[Q_Z] * mult);
    }

    /**
      * Stores *this * factor into *this (multiplication of each coord).
      *
      * \sa operator*(const T&) const
      */
    Quaternion & operator *= ( const T & factor )
    {
        m_array[Q_W] *= factor;
        m_array[Q_X] *= factor;
        m_array[Q_Y] *= factor;
        m_array[Q_Z] *= factor;

        return *static_cast<Quaternion*>(this);
    }

    /**
      * Returns *this / factor (division of each coord).
      *
      * \return_by_value
      *
      * \sa operator/=(const T&), operator*(const T&) const
      */
    Quaternion operator / ( const T & factor ) const
    {
        return Quaternion( m_array[Q_W] / mult, m_array[Q_X] / mult,  m_array[Q_Y] / mult,  m_array[Q_Z] / mult);
    }

    /**
      * Returns the array of the quaternion, as constant.
      *
      * \sa array(), operator()(int) const, operator[](int) const
      */
    const T * array() const
    {
        return static_cast<const Quaternion*>(this)->m_array;
    }

    /**
      * @returns the array of the quaternion, as non-constant.
      *
      * \sa array() const, operator()(int), operator[](int)
      */
    T * array()
    {
        return static_cast<Quaternion*>(this)->m_array;
    }
    /**
      * @returns a constant reference to the i-th coord of the vector.
      *
      * Same as operator[].
      *
      * \sa operator()(int), operator[](int) const
      */
    const T & operator () ( int i ) const
    {
        assert( i >= 0 && i < 4 );
        return array() [i];
    }

    /**
      * @returns a non-constant reference to the i-th coord of the vector.
      *
      * Same as operator[].
      *
      * \sa operator()(int) const, operator[](int)
      */
    T & operator () ( int i )
    {
        assert( i >= 0 && i < 4 );
        return array() [i];
    }

    /**
      * @returns a constant reference to the i-th coord of the vector.
      *
      * Same as operator().
      *
      * \sa operator[](int), operator()(int) const
      */
    const T & operator [] ( int i ) const
    {
        assert( i >= 0 && i < 4 );
        return array() [i];
    }

    /**
      * @returns a non-constant reference to the i-th coord of the vector.
      *
      * Same as operator().
      *
      * \sa operator[](int) const, operator()(int)
      */
    T & operator [] ( int i )
    {
        assert( i >= 0 && i < 4 );
        return array() [i];
    }
    /**
      * Returns a reference to the first coord of *this.
      *
      * \sa x() const
      */
    T & x() { return (*this)[0]; }

    /**
      * Returns a constant reference to the first coord of *this.
      *
      * \sa x()
      */
    const T & x() const { return (*this)[0]; }

    /**
      * Returns a reference to the second coord of *this.
      *
      * \sa y() const
      */
    T & y()
    {
        return (*this)[1];
    }

    /**
      * Returns a constant reference to the second coord of *this.
      *
      * \sa y()
      */
    const T & y() const
    {
        return (*this)[1];
    }

    /**
      * Returns a reference to the third coord of *this.
      *
      * \sa z() const
      */
    T & z()
    {
        return (*this)[2];
    }

    /**
      * Returns a constant reference to the third coord of *this.
      *
      * \sa z()
      */
    const T & z() const
    {
        return (*this)[2];
    }

    /**
      * Returns a reference to the fourth coord of *this.
      *
      * \sa w() const
      */
    T & w()
    {
        return (*this)[3];
    }

    /**
      * Returns a constant reference to the fourth coord of *this.
      *
      * \sa w()
      */
    const T & w() const
    {
        return (*this)[3];
    }

    void normalize();
    {
        *this *= 1.0f / sqrtf(quatNorm);
    }

    Quaternion inverse() const
    {
        Quaternion  inverse( m_array[Q_W], -m_array[Q_X], -m_array[Q_Y], -m_array[Q_Z] );
        inverse.normalize();

        return inverse;
    }

    void createFromEuler(float pitch, float yaw, float roll)
    {
        T cX, cY, cZ, sX, sY, sZ, cYcZ, sYsZ, sYcZ, cYsZ;

        pitch *= 0.5f;
        yaw   *= 0.5f;
        roll  *= 0.5f;

        cX = cosf(pitch);
        cY = cosf(yaw);
        cZ = cosf(roll);

        sX = sinf(pitch);
        sY = sinf(yaw);
        sZ = sinf(roll);

        cYcZ = cY * cZ;
        sYsZ = sY * sZ;
        sYcZ = sY * cZ;
        cYsZ = cY * sZ;

        m_array[Q_W] = cX * cYcZ + sX * sYsZ;
        m_array[Q_X] = sX * cYcZ - cX * sYsZ;
        m_array[Q_Y] = cX * sYcZ + sX * cYsZ;
        m_array[Q_Z] = cX * cYsZ - sX * sYcZ;
    }

    void rotateAroundAxis(const Quaternion &q)
    {
        T w, x, y, z;

        w = + m_array[Q_X] * q.m_array[Q_X] + m_array[Q_Y] * q.m_array[Q_Y] + m_array[Q_Z] * q.m_array[Q_Z];
        x = + m_array[Q_X] * q.m_array[Q_W] - m_array[Q_Y] * q.m_array[Q_Z] + m_array[Q_Z] * q.m_array[Q_Y];
        y = + m_array[Q_X] * q.m_array[Q_Z] + m_array[Q_Y] * q.m_array[Q_W] - m_array[Q_Z] * q.m_array[Q_X];
        z = - m_array[Q_X] * q.m_array[Q_Y] + m_array[Q_Y] * q.m_array[Q_X] + m_array[Q_Z] * q.m_array[Q_W];

        m_array[Q_W] = q.m_array[Q_W] * w - q.m_array[Q_X] * x - q.m_array[Q_Y] * y - q.m_array[Q_Z] * z;
        m_array[Q_X] = q.m_array[Q_W] * x + q.m_array[Q_X] * w + q.m_array[Q_Y] * z - q.m_array[Q_Z] * y;
        m_array[Q_Y] = q.m_array[Q_W] * y - q.m_array[Q_X] * z + q.m_array[Q_Y] * w + q.m_array[Q_Z] * x;
        m_array[Q_Z] = q.m_array[Q_W] * z + q.m_array[Q_X] * y - q.m_array[Q_Y] * x + q.m_array[Q_Z] * w;
    }

    void slerp(const Quaternion q1, const Quaternion q2, float t)
    {
        T p1, p2;

        T cosAlpha = ( q1.m_array[Q_X]*q2.m_array[Q_X] + q1.m_array[Q_Y]*q2.m_array[Q_Y]
                        + q1.m_array[Q_Z]*q2.m_array[Q_Z] + q1.m_array[Q_W]*q2.m_array[Q_W] );
        float alpha    = acosf(cosAlpha);
        T sinAlpha = sinf(alpha);

        if( sinAlpha > 0.0f ) {
            p1 = sinf( (1.0f-t)*alpha ) / sinAlpha;
            p2 = sinf( t*alpha ) / sinAlpha;
        } else {
            // both Quaternions are equal
            p1 = 1.0f;
            p2 = 0.0f;
        }

        m_array[Q_X] = p1*q1.m_array[Q_X] + p2*q2.m_array[Q_X];
        m_array[Q_Y] = p1*q1.m_array[Q_Y] + p2*q2.m_array[Q_Y];
        m_array[Q_Z] = p1*q1.m_array[Q_Z] + p2*q2.m_array[Q_Z];
        m_array[Q_W] = p1*q1.m_array[Q_W] + p2*q2.m_array[Q_W];
    }

    void getSpherical(float &alpha, float &beta) const // Geo: lon, lat
    {
        T  y = m_array[Q_Y];
        if ( y > 1.0f )
            y = 1.0f;
        else if ( y < -1.0f )
            y = -1.0f;
        beta = -asinf( y );
    
        if(m_array[Q_X] * m_array[Q_X] + m_array[Q_Z] * m_array[Q_Z] > 0.00005f) 
            alpha = -atan2f(m_array[Q_X], m_array[Q_Z]);
        else
            alpha = 0.0f;
    }

    void toMatrix(Matrix<T,4> &m)
    {
        T xy = m_array[Q_X] * m_array[Q_Y], xz = m_array[Q_X] * m_array[Q_Z];
        T yy = m_array[Q_Y] * m_array[Q_Y], yw = m_array[Q_Y] * m_array[Q_W];
        T zw = m_array[Q_Z] * m_array[Q_W], zz = m_array[Q_Z] * m_array[Q_Z];

        m[0][0] = 1.0f - 2.0f * (yy + zz);
        m[0][1] = 2.0f * (xy + zw);
        m[0][2] = 2.0f * (xz - yw);
        m[0][3] = 0.0f;

        T xx = m_array[Q_X] * m_array[Q_X], xw = m_array[Q_X] * m_array[Q_W], yz = m_array[Q_Y] * m_array[Q_Z];

        m[1][0] = 2.0f * (xy - zw);
        m[1][1] = 1.0f - 2.0f * (xx + zz);
        m[1][2] = 2.0f * (yz + xw);
        m[1][3] = 0.0f;

        m[2][0] = 2.0f * (xz + yw);
        m[2][1] = 2.0f * (yz - xw);
        m[2][2] = 1.0f - 2.0f * (xx + yy);
        m[2][3] = 0.0f;

        m[3][0] = 0.0f;
        m[3][1] = 0.0f;
        m[3][2] = 0.0f;
        m[3][3] = 1.0f;
    }

    void rotateAroundAxis(const Matrix<T,4> &m)
    {
        T x, y, z;

        x =  m[0][0] * m_array[Q_X] + m[1][0] * m_array[Q_Y] + m[2][0] * m_array[Q_Z];
        y =  m[0][1] * m_array[Q_X] + m[1][1] * m_array[Q_Y] + m[2][1] * m_array[Q_Z];
        z =  m[0][2] * m_array[Q_X] + m[1][2] * m_array[Q_Y] + m[2][2] * m_array[Q_Z];

        m_array[Q_W] = 1.0f; m_array[Q_X] = x; m_array[Q_Y] = y; m_array[Q_Z] = z;
    }

protected:

    /**
      * The quaternions's array of coordinates.
      */
    T    m_array[4];

    enum
    {
        Q_X = 0,
        Q_Y = 1,
        Q_Z = 2,
        Q_W = 3
    };
};

/**
  * Returns factor * v (multiplication of each coord of v by factor).
  */
template<typename T, typename Quaternion>
Quaternion operator *
( const T & factor, const Quaternion<T> & q )
{
    return q * factor;
}

/**
  * Allows to print a quaternion by simply doing
  * @code
    cout << myquaternion << endl;
  * @endcode
  */
template<typename T, typename Quaternion>
std::ostream & operator <<
( std::ostream & s, const Quaternion<T> & q )
{
    s << "(" << q(0) << ", " << q(1) << ", " << q( 2 ) << ", " << q( 3 ) << ")";
    return s;
}

typedef Quaternion<double> Quaterniond;
typedef Quaternion<float> Quaternionf;

}
#endif // EIGEN_QUATERNION_H
