/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_DAEMON_POLICYENGINE_H
#define DECIBEL_DAEMON_POLICYENGINE_H

#include <QtCore/QObject>
#include <QtDBus/QDBusObjectPath>

namespace QtTapioca
{
    class Connection;
    class Channel;
} // namespace QtTapioca

class ComponentManager;
class PolicyEnginePrivate;

/**
 * @brief This class implements the policy logic used to decide
 * which ChannelHandler to start in response to an incoming channel.
 *
 * This class implements the policy logic used to decide
 * which ChannelHandler to start in response to an incoming channel.
 * This decision is based on data provided by the ComponentManager.
 *
 * This class is not accessible via D-Bus.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class PolicyEngine : public QObject
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     * @param component_mgr Pointer to the ComponentManager.
     * @param parent The parent object.
     */
    explicit PolicyEngine(const ComponentManager * component_mgr,
                          QObject * parent = 0);
    /** Destructor */
    ~PolicyEngine();

public Q_SLOTS:
    /**
     * @brief React to incoming channels.
     * @param connection A pointer to the Connection the incoming channel
     * is part of.
     * @param channel A pointer to the incoming Channel.
     * @param request_cookie The request cookie. This is 0 if the channel
     *                       was opened by the remote side.
     *
     * This method triggers decides what to do with a incoming
     * information channel and triggers external ChannelHandlers
     * via D-Bus activation if required.
     */
    void onChannelOpened(QtTapioca::Connection * connection,
                         QtTapioca::Channel * channel,
                         quint64 request_cookie) const;

    /**
     * @brief Register a new ChannelHandler.
     * @param service The service name of the new ChannelHandler.
     * @param path The object path of the new ChannelHandler.
     * @param request_cookie The request cookie the new ChannelHandler
     *                       should handle.
     *
     * This method registers a new ChannelHandler which will get triggered
     * for the channel with the given request id.
     */
    void addRequestChannelHandler(const QString & service,
                                  const QDBusObjectPath & path,
                                  const quint64 & request_cookie);


private:
    PolicyEnginePrivate * const d;
};

#endif
