/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *  @author George Goldberg <grundleborg@googlemail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_DAEMON_DECIBELMAIN_H
#define DECIBEL_DAEMON_DECIBELMAIN_H

#include <QtCore/QObject>
#include <QtCore/QList>

class AccountConnectorBase;
class ContactConnectorBase;
class AccountManager;
class ContactManager;
class ProtocolManager;
class ConnectionFacade;
class ComponentManager;
class PolicyEngine;

/**
 * @brief Main class that handles the set up of the Decibel application.
 * @author George Goldberg <grundleborg@googlemail.com>
 *
 * A single instance of this class is created by the application's main()
 * function. It is then used to handle the asynchronous initialization of
 * the Decibel application.
 */
class DecibelMain : public QObject
{
    Q_OBJECT

public:
    /**
     * @brief Constructs a new DecibelMain object.
     * @param parent A pointer to the parent object.
     */
    DecibelMain(QObject * parent);
    /** @brief Destroys the DecibelMain Object. */
    ~DecibelMain();

public Q_SLOTS:
    /**
     * @brief Start the initialization of the application.
     *
     * This is the first slot called when setting up the application. It starts
     * the process of initializing decibel.
     */
    void startSettingUp();
    /**
     * @brief Called when the AccountConnector is set up.
     * @param success Indicates if the AccountConnector setup was successful.
     *
     * When the AccountConnector has been set up, this slot is called. If the
     * setup was successful, it starts setting up the ContactConnector.
     */
    void accountConnectorSetUp(bool success);
    /**
     * @brief Called when the ContactConnector is set up.
     * @param success Indicates if the ContactConnector setup was successful.
     *
     * When the ContactConnector has been set up, this slot is called. If the
     * setup was successful, it continues doing all the other initialization
     * that is needed to get decibel running.
     */
    void contactConnectorSetUp(bool success);

private:
    /** @brief Load all the decibel plugins and store them in lists. */
    void loadPlugins();
    /** @brief A pointer to the application-wide AccountConnector instance. */
    AccountConnectorBase * m_accountConnector;
    /** @brief A pointer to the application-wide AccountManager instance. */
    AccountManager * m_accountManager;
    /** @brief A pointer to the application-wide ComponentManager instance. */
    ComponentManager * m_componentManager;
    /** @brief A pointer to the application-wide ConnectionFacade instance. */
    ConnectionFacade * m_connectionFacade;
    /** @brief A pointer to the application-wide ContactConnector instance. */
    ContactConnectorBase * m_contactConnector;
    /** @brief A pointer to the application-wide ContactManager instance. */
    ContactManager * m_contactManager;
    /** @brief A pointer to the application-wide PolicyEngine instance. */
    PolicyEngine * m_policyEngine;
    /** @brief A pointer to the application-wide ProtocolManager instance. */
    ProtocolManager * m_protocolManager;
    /** 
     * @brief A list of pointers to all the plugins that implement the
     *        AccountConnectorBase interface.
     */
    QList<AccountConnectorBase*> m_accountConnectorPlugins;
    /** 
     * @brief A list of pointers to all the plugins that implement the
     *        ContactConnectorBase interface.
     */
    QList<ContactConnectorBase*> m_contactConnectorPlugins;
    
};

#endif  //Header guard
