/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *  @author George Goldberg <grundleborg@googlemail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "decibelmain.h"

#include "config.h"
#include "accountmanager.h"
#include "componentmanager.h"
#include "connectionfacade.h"
#include "contactmanager.h"
#include "policyengine.h"
#include "protocolmanager.h"
#include <accountconnectorbase.h>
#include <contactconnectorbase.h>

#include <Decibel/DBusNames>

#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMetaType>
#include <QtCore/QDebug>
#include <QtCore/QDir>
#include <QtCore/QString>
#include <QtCore/QPluginLoader>

DecibelMain::DecibelMain(QObject * parent)
 : QObject(parent),
   m_accountConnector(0),
   m_accountManager(0),
   m_componentManager(0),
   m_connectionFacade(0),
   m_contactConnector(0),
   m_contactManager(0),
   m_policyEngine(0),
   m_protocolManager(0)
{ }

DecibelMain::~DecibelMain()
{
    if(m_accountConnector != 0)
    { m_accountConnector->closeStorage(); }
    delete m_accountConnector;
    delete m_contactConnector;
    delete m_protocolManager;
    delete m_connectionFacade;
    delete m_contactManager;
    delete m_accountManager;
    delete m_componentManager;
    delete m_policyEngine;
}

void
DecibelMain::startSettingUp()
{
    /* First, we load all the plugins. */
    loadPlugins();

    qDebug() << "Setting Up Account Connector...";
    if(!m_accountConnectorPlugins.isEmpty())
    {
        // FIXME: How do we choose if more than one is available?
        m_accountConnector = m_accountConnectorPlugins.at(0);
        /* Connect its accountDataAvailable to the next slot. */
        connect(m_accountConnector, SIGNAL(accountDataAvailable(bool)),
                this, SLOT(accountConnectorSetUp(bool)));
        /* Tell the AccountConnector to set itself up. */
        m_accountConnector->openStorage();
        return;
     }
     qDebug() << "No Account Connector Found.";
}

void
DecibelMain::loadPlugins()
{
    qDebug() << "Loading Plugins..";
    QDir pluginsDir = QDir(PLUGIN_DIR);     //PLUGIN_DIR is defined in config.h
    foreach (QString fileName, pluginsDir.entryList(QDir::Files))
    {
        QPluginLoader loader(pluginsDir.absoluteFilePath(fileName));
        QObject *plugin = loader.instance();

        if (!plugin || !loader.isLoaded())
        {
            qWarning("Unable to load plugin \"%s\": %s", qPrintable(fileName), qPrintable(loader.errorString()));
            continue;
        }

        AccountConnectorBase *accountConnector = qobject_cast<AccountConnectorBase*>(plugin);
        if (accountConnector != 0)
        {
            /*
             * we have a AccountConnector. Add it to the list
             */
            m_accountConnectorPlugins.append(accountConnector);
        }

        ContactConnectorBase *contactConnector = qobject_cast<ContactConnectorBase*>(plugin);
        if(contactConnector != 0)
        {
            /*
             * We have a ContactConnector. Add it to the list.
             */
            m_contactConnectorPlugins.append(contactConnector);
        }
     }
}
void
DecibelMain::accountConnectorSetUp(bool success)
{
    /*
     * If account connector was not set up successfully,
     * we print a warning and abort initialization.
     */
    if(!success)
    {
        qDebug() << "Setting up account connector failed.";
        return;
    }
    qDebug() << "Account Connector set up OK."
             << "Setting up Contact Connector...";
    /*
     * Account Connector set up successfully.
     * Now we set up the ContactConnector.
     */
    if (!m_contactConnectorPlugins.isEmpty())
    {
        /* We have a ContactConnector. */
        // FIXME: How do we choose if more than one is available?
        m_contactConnector = m_contactConnectorPlugins.at(0);
        /* Connect its contactDataAvailable to the next slot. */
        connect(m_contactConnector, SIGNAL(contactDataAvailable(bool)),
                this, SLOT(contactConnectorSetUp(bool)));
        /* Tell the ContactConnector to set itself up. */
        m_contactConnector->openStorage();
        return;
     }
     qDebug() << "No Contact Connector Found.";
}

void
DecibelMain::contactConnectorSetUp(bool success)
{
    /*
     * If contact connector was not set up successfully,
     * we print a warning and abort initialization.
     */
    if(!success)
    {
        qDebug() << "Setting up contact connector failed.";
        return;
    }
    qDebug() << "Contact Connector set up OK."
             << "Setting up the rest of Decibel...";
    /*
     * ContactConnector set up successfully.
     * Now we continue and set up everything else.
     */
    m_protocolManager = new ProtocolManager(this);
    m_connectionFacade = new ConnectionFacade(m_protocolManager, this);
    m_accountManager = new AccountManager(m_connectionFacade, m_accountConnector, this);
    m_componentManager = new ComponentManager(this);
    m_policyEngine = new PolicyEngine(m_componentManager, this);
    m_contactManager = new ContactManager(m_accountManager, m_contactConnector, this);

    QObject::connect(m_connectionFacade, SIGNAL(shuttingDown()),
                     m_accountManager, SLOT(doBlockReconnects()));
    QObject::connect(m_connectionFacade, SIGNAL(channelOpened(QtTapioca::Connection *,QtTapioca::Channel *,quint64)),
                     m_policyEngine, SLOT(onChannelOpened(QtTapioca::Connection *,QtTapioca::Channel *,quint64)));
    QObject::connect(m_contactManager, SIGNAL(channelOpened(QtTapioca::Connection *,QtTapioca::Channel *,quint64)),
                     m_policyEngine, SLOT(onChannelOpened(QtTapioca::Connection *,QtTapioca::Channel *,quint64)));
    QObject::connect(m_connectionFacade, SIGNAL(connectionOpened(QtTapioca::Connection *)),
                     m_contactManager, SLOT(onConnectionOpened(QtTapioca::Connection *)));
    QObject::connect(m_connectionFacade, SIGNAL(connectionClosed(QtTapioca::Connection *)),
                     m_contactManager, SLOT(onConnectionClosed(QtTapioca::Connection *)));
    QObject::connect(m_contactManager, SIGNAL(registerRequestChannelHandler(QString,QDBusObjectPath,quint64)),
                     m_policyEngine, SLOT(addRequestChannelHandler(QString,QDBusObjectPath,quint64)));
    connect(m_connectionFacade, SIGNAL(connectionOpened(QtTapioca::Connection *)),
            m_accountManager, SLOT(onConnectionOpened(QtTapioca::Connection *)));

    // Close connections while we still have an eventloop.
    QObject::connect(this->parent(), SIGNAL(aboutToQuit()),
                     m_connectionFacade, SLOT(onAboutToQuit()));

    // register on D-BUS:
    QDBusConnection::sessionBus().registerService(Decibel::daemon_service);
    qDebug() << "Service" << Decibel::daemon_service << "registered with session bus.";
    QDBusConnection::sessionBus().registerObject(Decibel::daemon_protocolmanager_path,
                                                 m_protocolManager);
    QDBusConnection::sessionBus().registerObject(Decibel::daemon_accountmanager_path,
                                                 m_accountManager);
    QDBusConnection::sessionBus().registerObject(Decibel::daemon_componentmanager_path,
                                                 m_componentManager);
    QDBusConnection::sessionBus().registerObject(Decibel::daemon_contactmanager_path,
                                                 m_contactManager);
    qDebug() << "Set up complete. Decibel is ready to go...";
}
