/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *  @author George Goldberg <grundleborg@googlemail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_DAEMON_CONTACTSTATEMACHINE_H
#define DECIBEL_DAEMON_CONTACTSTATEMACHINE_H

#include <QtCore/QObject>
#include <QtDBus/QDBusContext>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtCore/QHash>
#include <QtCore/QList>

#include <Decibel/Types>

class ContactConnectorBase;
class ContactManager;
class AccountManager;

namespace QtTapioca {
    class Connection;
    class Contact;
}

/**
 * @brief The ContactStateMachine is a class that carries out connecting
 *        to a contact.
 * @author George Goldberg <grundleborg@googlemail.com>
 *
 * An instance of the ContactStateMachine class is created for every request
 * to connect to a contact/url received via the ContactManager. It carries
 * out setting up the connection via asynchronous communication with the
 * ContactConnector. All signals recived from the ContactConnector are relayed
 * via the ContactManagerPrivate class.
 */
class ContactStateMachine : public QObject
{
    Q_OBJECT

public:
    /**
     * @brief Create a new ContactStateMachine.
     * @param cookie The channel cookie for the channel this ContactStateMachine
     *        will set up.
     * @param connector A pointer to the ContactConnector to be used for looking
     *        up contact data.
     * @param accountManager A pointer to the AccountManager to be used for
     *        local account data.
     * @param parent A pointer to the parent ContactManager object.
     * @param external2telepathy A pointer to the ContactManager's mapping
     *        between external and telepathy contact IDs.
     * @param telepathy2external A pointer to the ContactManager's mapping
     *        between telepathy and external contact IDs.
     */
    ContactStateMachine(const quint64 cookie,
                        ContactConnectorBase *connector,
                        AccountManager *accountManager,
                        ContactManager *parent,
                        QHash<QString, QtTapioca::Contact *> * external2telepathy,
                        QHash<QtTapioca::Contact *, QString> * telepathy2external);
    /** @brief Destructor... */
    ~ContactStateMachine();

    /**
     * @brief Slot called when a contact exists for a specific request.
     * @param got Does the contact exists in the contact connector?
     *
     * This slot is called with the response to a call to
     * ContactConnector::gotContact().
     */
    void gotContact(bool got);
    /**
     * @brief Slot called with the URIs corresponding to a contact for a
     * specific request.
     * @param uris The URIs for the requested contact.
     *
     * This slot is called with the response to a call to
     * ContactConnector::getURIs().
     */
    void gotUris(QStringList uris);
    /**
     * @brief Slot called when the contact a URI belongs to is found for a
     * specific request.
     * @param external_id The ID of the contact of which the specified URI forms
     * a part.
     *
     * This slot is called with the response to a call to
     * ContactConnector::findURI().
     */
    void uriFound(const QString & external_id);
    /**
     * @brief Slot called when a contact is added to the ContactConnector for a
     * specific request.
     * @param contact_url The ID of the newly added contact as seen in the address
     *                    book of the system.
     *
     * This slot is called with the response to a call to
     * ContactConnector::addContact().
     */
    void contactAdded(const QString & contact_url);

    /**
     * @brief Connect to a contact using a given account.
     * @param contact_url The url of the contact to connect to as seen in the
     *                    addressbook of the system.
     * @param account_handle A handle to the account to use.
     * @param type The type of connection requested.
     * @param service The service name that is to handle the newly created
     *                channel.
     * @param path The path to the object that is to handle the newly created
     *              channel.
     *
     * This initiates the creation of a channel using the parameters given.
     */
    void contactContactUsingAccount(const QString & contact_url,
                                    const int account_handle,
                                    const int type,
                                    const QString & service,
                                    const QDBusObjectPath & path);
    /**
     * @brief Connect to a somebody using a given url.
     * @param contact_uri The URL of the contact to connect to.
     * @param account_handle A handle to the account to use.
     * @param type The type of connection requested.
     * @param service The service name that is to handle the newly created
     *                channel.
     * @param path The path to the object that is to handle the newly created
     *              channel.
     *
     * This initiates the creation of a channel using the parameters given.
     */
    void contactUrlUsingAccount(const QString & contact_uri,
                                const int account_handle,
                                const int type,
                                const QString & service,
                                const QDBusObjectPath & path);
    /**
     * @brief Connect to a contact URL using any account available.
     * @param contact_uri The URL of the contact to connect to.
     * @param type The type of connection requested.
     * @param service The service name that is to handle the newly created
     *                channel.
     * @param path The path to the object that is to handle the newly created
     *              channel.
     *
     * This initiates the creation of a channel using the parameters given.
     */
    void contactUrl(const QString & contact_uri,
                    const int type,
                    const QString & service,
                    const QDBusObjectPath & path);
    /**
     * @brief An enumeration of all the possible states the
     * ContactStateMachine can be in.
     */
    enum State {
        InitialState,
        CheckContactExistsState,
        GettingUrisState,
        FindingUriState,
        AddingContactState,
        ContactingContactState,
        CompletedState,
        ErrorState
    };

Q_SIGNALS:
    /**
     * @brief Indicates that the channel creation failed.
     * @param type The type of error to occur.
     * @param msg The text of the error message.
     */
    void sendErrorReply(const QString & type, const QString & msg);

    /**
     * @brief Indicates that connecting to the cotnact succeeded.
     * @param connection The connection to the contact.
     * @param channel The channel to that contact.
     * @param cookie The channel request cookie.
     */
    void contactConnected(QtTapioca::Connection * const connection,
                          QtTapioca::Channel * const channel,
                          const quint64 cookie);
    /**
     * @brief Indicates that a new connection has been created that should be
     *        appended to the ContactManagerPrivate's list of connections.
     * @param connection A pointer to the newly created connection.
     * @param contact A pointer to the contact with whom the connection is.
     */
    void connectionListAppend(QtTapioca::Connection * connection,
                              QtTapioca::Contact * contact);
    /**
     * @brief Signals that a particular channel handler should be set to take
     *        responsibility for this channel when it is created.
     * @param service The DBus service for the channel handler.
     * @param path The object path to the channel handler.
     * @param cookie The channel cookie for the channel that this ChannelHandler
     *        is being registered to take responsibility for.
     */
    void registerRequestChannelHandler(const QString & service,
                                       const QDBusObjectPath & path,
                                       const quint64 & cookie);

private:
    QtTapioca::Connection * connectAccount(const int account_handle);
    void registerContact();
    bool testUri(const QString & uri);
    QString getProtocol(const QString & uri);
    QtTapioca::Contact * mapFromExternal(const QString & external_url);
    QString getCMInternalUri(const QString & uri);
    QList<uint> findAccountsSupporting(const QString & protocol);
    void contactContactViaConnection();

    /** @brief A pointer to the ContactConnector. */
    ContactConnectorBase *m_connector;
    /** @brief A pointer to the ContactManager. */
    ContactManager *m_contactManager;
    /** @brief A pointer to the AccountManager. */
    AccountManager *m_accountManager;

    /**
     * @brief A pointer to the ContactManager's mapping between external and
     * telepathy contacts.
     */
    QHash<QString, QtTapioca::Contact *> * m_external2telepathy;
    /**
     * @brief A pointer to the ContactManager's mapping between telepathy and
     * external contacts.
     */
    QHash<QtTapioca::Contact *, QString> * m_telepathy2external;

    /** @brief The current state of the ContactStateMachine. */
    State m_state;

    /** @brief The contact id of the contact we are opening a connection with */
    QString m_contact_url;
    /**
     * @brief The account handle of the local account we are
     * connecting through.
     */
    int m_account_handle;
    /** @brief The type of connection we are setting up. */
    int m_type;
    /** @brief The service name that will handle the created channel. */
    QString m_service;
    /** @brief The object path that will handle the created channel. */
    QDBusObjectPath m_path;
    /** @brief The channel cookie for the channel that will be created. */
    const quint64 m_cookie;
    /** @brief The protocol for the connection being initiated. */
    QString m_protocol;
    /** @brief The URI of the recipient of this connection. */
    QString m_uri;

    /** @brief The connection object for to be used. */
    QtTapioca::Connection * m_connection;
    /** @brief The contact the channel will be opened with. */
    QtTapioca::Contact * m_contact;
};

#endif  //header guard
