/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH <info@basyskom.de>
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *  @author George Goldberg <grundleborg@googlemail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, wr-ite to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "contactstatemachine.h"
#include "contactmanager.h"
#include "contactconnectorbase.h"
#include "accountmanager.h"

#include <Decibel/Errors>
#include <Decibel/AccountData>

#include <QtCore/QCoreApplication>
#include <QtCore/QDebug>

#include <QtTapioca/Connection>
#include <QtTapioca/ContactList>
#include <QtTapioca/Contact>
#include <QtTapioca/Channel>
#include <QtTapioca/PresenceState>

ContactStateMachine::ContactStateMachine(const quint64 cookie,
                                         ContactConnectorBase *connector,
                                         AccountManager *accountManager,
                                         ContactManager *parent,
                                         QHash<QString, QtTapioca::Contact *> *
                                             external2telepathy,
                                         QHash<QtTapioca::Contact *, QString> *
                                             telepathy2external) :
    m_connector(connector),
    m_contactManager(parent),
    m_accountManager(accountManager),
    m_external2telepathy(external2telepathy),
    m_telepathy2external(telepathy2external),
    m_state(ContactStateMachine::InitialState),
    m_cookie(cookie)
{
    // Make sure we have a sane initial state:
    Q_ASSERT(0 != m_cookie);
    Q_ASSERT(0 != m_connector);
    Q_ASSERT(0 != m_contactManager);
    Q_ASSERT(0 != m_accountManager);
    Q_ASSERT(0 != m_external2telepathy);
    Q_ASSERT(0 != m_telepathy2external);
}

ContactStateMachine::~ContactStateMachine()
{ /* Nothing to do here! */ }

void ContactStateMachine::contactContactUsingAccount(const QString & contact_url,
                                                     const int account_handle,
                                                     const int type,
                                                     const QString & service,
                                                     const QDBusObjectPath & path)
{
    Q_ASSERT(ContactStateMachine::InitialState == m_state);

    /*
     * We must store all the parameters
     * passed to this method, so they can
     * be accessed by other methods later.
     */
    m_contact_url = contact_url;
    m_account_handle = account_handle;
    m_type = type;
    m_service = service;
    m_path = path;

    /*
     * next, we must check whether the
     * contact really exists, and update
     * the state to say we're checking
     * that.
     */
    m_state = ContactStateMachine::CheckContactExistsState;
    m_connector->gotContact(contact_url, m_cookie);
}

void ContactStateMachine::contactUrlUsingAccount(const QString & contact_uri,
                                                 const int account_handle,
                                                 const int type,
                                                 const QString & service,
                                                 const QDBusObjectPath & path)
{
    Q_ASSERT(ContactStateMachine::InitialState == m_state);

    /*
     * We must store all the parameters
     * passed to this method, so they can
     * be accessed by other methods later.
     */
    m_uri = contact_uri;
    m_account_handle = account_handle;
    m_type = type;
    m_service = service;
    m_path = path;

    /*
     * next, we must check that the contact URI
     * we received is really valid. If not return
     * with an error.
     */
    qDebug() << "ContactStateMachine::contactUrlUsingAccount(): Checking URI is valid.";
    if (!testUri(contact_uri))
    {
        emit sendErrorReply(Decibel::ErrorNoSuchContact,
                            tr("Contact URI %1 not found.", "1: contact_uri").
                                arg(contact_uri));
        return;
    }

    /*
     * contact URI is valid. Now check that
     * local protocol and remote protocol are
     * the same, and if not, output an error.
     */
    // FIXME: Should we fail and emit an error if the protocols do not match?
    QString remote_protocol = getProtocol(contact_uri);

    QString local_protocol = m_accountManager->protocol(account_handle);
    if (local_protocol != remote_protocol)
    {
        qWarning() << "CtM: Local protocol" << local_protocol
                   << "and remote protocol" << remote_protocol
                   << "differ!";
    }
    /*
     * save remote_protocol as the protocol
     * member variable.
     */
    m_protocol = remote_protocol;

    /*
     * connect the account_handle we were given,
     * and emit an error and return if the
     * connecting fails.
     */
    m_connection = connectAccount(account_handle);
    if (0 == m_connection)
    {
        emit sendErrorReply(Decibel::ErrorNoSuchConnection,
                       tr("Account %1 not connected.", "1: account_handle").
                       arg(account_handle));
        return;
    }
    /*
     * connecting succeeded. Now we have
     * a connection from this point onwards.
     */
    Q_ASSERT(0 != m_connection);

    // Create contact:
    QString cm_internal_uri = getCMInternalUri(contact_uri);
    m_contact = m_connection->contactList()->
                                 contact(cm_internal_uri);
    if (0 == m_contact)
    { m_contact = m_connection->contactList()->addContact(cm_internal_uri); }

    m_state = ContactStateMachine::FindingUriState;
    registerContact();
}

void
ContactStateMachine::contactUrl(const QString & contact_uri,
                           const int type,
                           const QString & service,
                           const QDBusObjectPath & path)
{
    Q_ASSERT(ContactStateMachine::InitialState == m_state);

    /*
     * We must store all the parameters
     * passed to this method, so they can
     * be accessed by other methods later.
     */
    m_uri = contact_uri;
    m_type = type;
    m_service = service;
    m_path = path;

    /*
     * next, we must check that the contact URI
     * we received is really valid. If not return
     * with an error.
     */
    if (!testUri(contact_uri))
    {
        emit sendErrorReply(Decibel::ErrorNoSuchContact,
                       tr("Contact URI %1 not found.", "1: contact_uri").
                       arg(contact_uri));
        return;
    }

    m_protocol = getProtocol(contact_uri);

    // Find accounts supporting protocol:
    QList<uint> possibleAccounts = findAccountsSupporting(m_protocol);

    if (possibleAccounts.isEmpty())
    {
        emit sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("No account found that supports protocol %1.",
                       "1: protocol").
                       arg(m_protocol));
        return;
    }

    /*
     * now we leave the state at InitialState, and proceed
     * by calling contactUrlUsingAccount();
     */
    // FIXME: how do we choose which account to use when more than 1 available?
    contactUrlUsingAccount(contact_uri, possibleAccounts[0], type,
                                  service, path);
}

void
ContactStateMachine::gotContact(bool got_contact)
{
    //Check state
    Q_ASSERT(ContactStateMachine::CheckContactExistsState == m_state);
    //Check all the member variables that we assume are set.
    Q_ASSERT(m_account_handle != 0);
    Q_ASSERT(!m_contact_url.isEmpty());
    /*
     * state is appropriate. Now check
     * if the contact was found.
     */
    if (!got_contact)
    {
        /*
         * contact has not been found.
         * change state and signal
         * dbus error reply and return.
         */
        m_state = ContactStateMachine::ErrorState;

        emit sendErrorReply(Decibel::ErrorNoSuchConnection,
                    tr("Account %1 not connected.", "1: account_handle").
                    arg(m_account_handle));
        return;
    }

    /*
     * test the account handle.
     */
    m_connection = connectAccount(m_account_handle);
    if (0 == m_connection)
    {
        /*
         * connection doesn't exist.
         * change state and emit
         * error signal and return.
         */
        m_state = ContactStateMachine::ErrorState;

        emit sendErrorReply(Decibel::ErrorNoSuchConnection,
                    tr("Account %1 not connected.", "1: account_handle").
                    arg(m_account_handle));
        return;
    }

    Q_ASSERT(0 != m_connection);

    /*
     * check if the contact id's are already known.
     */
    m_contact = mapFromExternal(m_contact_url);

    /*
     * if not, then we need to get the URI's for the contact,
     * and update its state.
     */
    if (0 == m_contact)
    {
        QString protocol(m_accountManager->protocol(m_account_handle));
        m_protocol = protocol;
        m_state = ContactStateMachine::GettingUrisState;
        m_connector->getURIs(m_contact_url, protocol, m_cookie);

        return;
    }

    Q_ASSERT(0 != m_contact);

    /*
     * contact ids are in Map already, so we can call the
     * next method directly after updating state and
     * saving the new local variables.
     */
    m_state = ContactStateMachine::ContactingContactState;
    contactContactViaConnection();
}

QtTapioca::Connection *
ContactStateMachine::connectAccount(const int account_handle)
{
    // check Account:
    if (!m_accountManager->gotAccount(account_handle))
    {
        qWarning() << "CtM: Account" << account_handle << "is invalid.";
        return 0;
    }

    // Bring up account (if not already up):
    if (Decibel::isOffline(m_accountManager->presence(account_handle).name()))  // FIXME: PresenceState
    {
        Q_ASSERT(0 == m_accountManager->connectionOf(account_handle));
        m_accountManager->setPresence(account_handle,
                                      QtTapioca::PresenceState(Decibel::state_available, QtTapioca::PresenceState::AvailableType));  // FIXME: PresenceState
    }

    // Wait for connection to get established...
    // FIXME: Should this be changed to be async, now everything else is?
    QtTapioca::Connection * connection(0);
    while (0 == connection)
    {
        QCoreApplication::instance()->processEvents();
        connection = m_accountManager->connectionOf(account_handle);
    }

    return connection;
}

void
ContactStateMachine::gotUris(QStringList uris)
{
    /*
     * check we are in a state where we know
     * what to do with this method.
     */
    Q_ASSERT(m_state == ContactStateMachine::GettingUrisState);
    //Check all member vars that we assume to be set.
    Q_ASSERT(!m_contact_url.isEmpty());
    Q_ASSERT(!m_protocol.isEmpty());
    Q_ASSERT(m_connection != 0);
    /*
     * we know what to do in this state. Check that
     * the uris list is not empty.
     */
    if (uris.isEmpty())
    {
        m_state = ContactStateMachine::ErrorState;
        emit sendErrorReply(Decibel::ErrorContactDataIncomplete,
                        tr("Contact id %1 does not have data defined for protocol %2.",
                        "1: contact_id 2: protocol").
                        arg(m_contact_url).arg(m_protocol));
        return;
    }
    /*
     * add the contact to the tapioca connection.
     */
    // FIXME: Handle multiple uris...
    m_contact = m_connection->contactList()->addContact(getCMInternalUri(uris[0]));

    /*
     * save new local variables, before
     * registering the contact.
     */
    m_state = ContactStateMachine::FindingUriState;
    registerContact();

    return;
}

void
ContactStateMachine::registerContact()
{
    //Check all member vars we assume to be set.
    Q_ASSERT(0 != m_connection);
    Q_ASSERT(0 != m_contact);
    Q_ASSERT(!m_protocol.isEmpty());

    QString uri(m_contact->uri());
    QString uri_protocol(getProtocol(uri));
    if (uri_protocol.isEmpty())
    { uri = m_protocol + QString("://") + uri; }
    else
    {
        if (m_protocol != uri_protocol)
        {
            qWarning() << "CtM: Fixing protocol in URI, was" << m_protocol
                        << "changed to" << uri_protocol;
            uri = uri_protocol + "://" + getCMInternalUri(uri);
        }
    }
    Q_ASSERT(uri.indexOf(QString("://") != 0));
    Q_ASSERT(getProtocol(uri) == m_protocol);
    Q_ASSERT(getCMInternalUri(uri) == getCMInternalUri(m_contact->uri()));

    // mark the handle as belonging to the given connection:
    emit connectionListAppend(m_connection, m_contact);

    /*
     * store the URI in this object as
     * a member variable for use in later
     * methods.
     */
    m_uri = uri;

    /*
     * check to see if the URI we are currently using
     * exists as a ContactAccount in the contactConnector
     * or not.
     */
    m_connector->findURI(uri, m_cookie);
}

void
ContactStateMachine::uriFound(const QString & external_url)
{
    /*
     * check we are in the appropriate state.
     */
    Q_ASSERT(m_state == ContactStateMachine::FindingUriState);
    //Check all member variables we assume are set.
    Q_ASSERT(!m_uri.isEmpty());
    Q_ASSERT(0 != m_contact);

    /*
     * check if the contact is already stored
     * in the external PIM database or not.
     */
    if (!external_url.isEmpty())
    {
        /*
         * contact is not already in PIM system. Add it
         */
        // FIXME: What data should we be adding to PIM system?
        qDebug() << "ContactStateMachine::uriFound(): contact not already in contactConnector, saving it there.";
        QVariantMap contact_data;
        contact_data["URI"] = m_uri;
        contact_data["Name"] = m_contact->alias();

        /*
         * update state and return.
         */
        m_state = ContactStateMachine::AddingContactState;

        m_connector->addContact(contact_data, m_cookie);
        // FIXME: the conact ID is not registered with telepathy2external and the other... is this a mistake?
        return;
    }
    /*
     * contact is already stored.
     * Update state then call next
     * method directly.
     */
    qDebug() << "ContactStateMachine::uriFound(): contact already in contactConnector. no need to resave. Should we update it?";
    // FIXME: should we update the contact database here?

    Q_ASSERT(0 != external_url);
    m_telepathy2external->insert(m_contact, external_url);
    m_external2telepathy->insert(external_url, m_contact);

    m_state = ContactStateMachine::ContactingContactState;
    contactContactViaConnection();
    return;
}

void
ContactStateMachine::contactAdded(const QString & contact_url)
{
    /*
     * Check we are in the appropriate state
     * continue.
     */
    Q_ASSERT(m_state == ContactStateMachine::AddingContactState);
    m_state = ContactStateMachine::ContactingContactState;
    m_contact_url = contact_url;
    contactContactViaConnection();
}

void
ContactStateMachine::contactContactViaConnection()
{
    /*
     * check we are in the appropriate
     * state before continuing.
     */
    Q_ASSERT(m_state == ContactStateMachine::ContactingContactState);
    /*
     * contact the contact, before
     * emitting the signal to say we
     * are done.
     */
    Q_ASSERT(0 != m_contact);
    Q_ASSERT(0 != m_connection);
    Q_ASSERT(m_type >= int(QtTapioca::Channel::Text) &&
                m_type <= int(QtTapioca::Channel::Stream));

    Q_ASSERT(!m_service.isEmpty());
    // FIXME: how do we assert check m_path?

    emit registerRequestChannelHandler(m_service, m_path, m_cookie);

    QtTapioca::Channel * channel =
        m_connection->createChannel(QtTapioca::Channel::Type(m_type), m_contact,
                                    false);
    if (0 == channel)
    {
        qWarning() << "CtM: Failed to create channel of type" << m_type
                << "to contact" << m_contact << ".";
        return;
    }
    qDebug() << "CtM: Got my channel!";

    /*
     * we've finished, so change state to completed.
     */
    m_state = ContactStateMachine::CompletedState;

    /*
     * emit a signal to the CM telling
     * it we're done.
     */
    emit contactConnected(m_connection, channel, m_cookie);
}

bool
ContactStateMachine::testUri(const QString & uri)
{
    // Make sure contact_uri is valid:
    QString remote_protocol = getProtocol(uri);
    if (uri.isEmpty())
    {
        qWarning() << "CtM: Contact URI is empty.";
        return false;
    }
    else if (remote_protocol.isEmpty())
    {
        qWarning() << "CtM: Contact URI" << uri
                << "does not name a protocol.";
        return false;
    }
    return true;
}

QString
ContactStateMachine::getProtocol(const QString & uri)
{
    int pos = uri.indexOf(QString("://"));
    if (pos < 0) { return QString(); }
    else { return uri.left(pos); }
}

/**
 * @brief Find all accounts supporting a given protocol.
 * @param protocol The protocol the account must support.
 * @return A list of accounts supporting the requested protocol.
 *
 * Find all the accounts that support a given protocol. Accounts in an
 * online state are returned first in the list.
 */
QList<uint>
ContactStateMachine::findAccountsSupporting(const QString & protocol)
{
    QList<uint> allAccounts(m_accountManager->listAccounts());
    QList<uint> possibleAccounts;
    QList<uint> possibleOnlineAccounts;
    uint currentAccount;
    foreach (currentAccount, allAccounts)
    {
        if (m_accountManager->protocol(currentAccount) == protocol)
        {
            if (0 != m_accountManager->connectionOf(currentAccount))
            { possibleOnlineAccounts.append(currentAccount); }
            else { possibleAccounts.append(currentAccount); }
        }
    }

    return possibleOnlineAccounts + possibleAccounts;
}

QString
ContactStateMachine::getCMInternalUri(const QString & uri)
{
    int pos = uri.indexOf(QString("://"));
    if (pos < 0) { return uri; }
    else { return uri.mid(pos + 3); }
}

/**
 * @brief Map a external contact ID to the internal Contact data.
 * @param external_id The contact's ID in the external PIM system.
 * @return A pointer to the internal Contact structure. This pointer is 0
 * if the contact is not know to telepathy.
 *
 * Map a external contact ID to the internal Contact data.
 */
QtTapioca::Contact *
ContactStateMachine::mapFromExternal(const QString & external_url)
{
    if (!m_external2telepathy->contains(external_url)) { return 0; }

    Q_ASSERT(m_external2telepathy->value(external_url) != 0);
    return m_external2telepathy->value(external_url);
}
