/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH <info@basyskom.de>
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *  @author George Goldberg <grundleborg@googlemail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_DAEMON_CONTACTMANAGER_P_H
#define DECIBEL_DAEMON_CONTACTMANAGER_P_H

#include <QtCore/QHash>
#include <QtCore/QPointer>

#include <QtTapioca/Contact>

class AccountManager;
class ContactManager;
class ContactManagerAdaptor;
class ContactConnectorBase;
class ContactStateMachine;

namespace QtTapioca
{
    class Connection;
    class ContactList;
    class Handle;
    class Channel;
} // namespace QtTapioca

/**
 * @brief Private data class for the ContactManager.
 * @author Tobias Hunger <info@basyskom.de>
 * @author George Goldberg <grundleborg@googlemail.com>
 */
class ContactManagerPrivate : public QObject
{
    Q_OBJECT

public:
    /** @brief Constructor. */
    ContactManagerPrivate(AccountManager * account_mgr,
                          ContactConnectorBase * connector,
                          ContactManager * contact_mgr);

    /** @brief Destructor. */
    ~ContactManagerPrivate();

    /**
     * @brief Deregister a contact.
     * @param connection A pointer to a connection.
     *
     * Deregister all contacts found on the given connection from the
     * mappings between internal and external contact IDs.
     */
    void deregisterContacts(const QtTapioca::Connection * connection);

    /**
     * @brief Update the presence of a contact.
     * @param internal_contact internal (telepathy) contact handle.
     * @param presence The presence state.
     * @param params The presence parameters.
     *
     * Update the presence state and message of a contact.
     */
    void setPresence(QtTapioca::Contact * internal_contact,
                     const QtTapioca::PresenceState & presence,
                     const QVariantMap & params);

    /**
     * @brief Map a Contact to its external ID.
     * @param contact A pointer to the Contact.
     * @return A empty string if the contact is not known extrnally and
     * the contact's ID otherwise.
     *
     * Map a Contact to its external ID.
     */
    QString mapToExternal(QtTapioca::Contact * contact);

    void registerContact(const QString & protocol,
                        QtTapioca::Contact * contact,
                        const QtTapioca::Connection * connection);

    /**
     * @brief Regester a account.
     * @param contact_url The url of the contact as used to address the addressbook.
     * @param cookie Request cookie
     *
     * FIXME: Provide better documentation.
     */
    void completeRegisterContact(const QString & contact_url, const quint64 cookie);

    /**
     * @brief NewChannelInfo struct.
     *
     * FIXME: Provide better documentation.
     */
    struct NewChannelInfo
    {
        /** @brief The connection of the new channel. */
        QtTapioca::Connection * connection;
        /** @brief The new channel itself. */
        QtTapioca::Channel * channel;
        /** @brief The cookie that requested the channel. */
        quint64 cookie;

        /** @brief Constructor */
        NewChannelInfo() : connection(0), channel(0), cookie(0) { }
        /** @brief Constructor */
        NewChannelInfo(QtTapioca::Connection * conn,
                       QtTapioca::Channel * chan,
                       const quint64 & keks) :
            connection(conn),
            channel(chan),
            cookie(keks)
        { }
        /** @brief Constructor */
        NewChannelInfo(const NewChannelInfo & info) :
            connection(info.connection),
            channel(info.channel),
            cookie(info.cookie)
        { }
        /** @brief Destructor */
        ~NewChannelInfo() { }
    };

    /** @brief A pointer to the AccountManager. */
    QPointer<AccountManager> accountManager;
    /** @brief A pointer to the D-Bus Adaptor of the ContactManager. */
    QPointer<ContactManagerAdaptor> adaptor;
    /** @brief A list containing all newly created channels. */
    QList<NewChannelInfo> newChannelList;

public Q_SLOTS:
    /** @brief Do Tapioca-Magic to set up the actual channel. */
    void contactContactViaConnection(QtTapioca::Connection * const connection,
                                     QtTapioca::Channel * const channel,
                                     const quint64 cookie);
    /**
     * @brief Append a new connection to the internal list of
     * new connections.
     * @param connection A pointer to the new connection.
     * @param contact A pointer to the contact being contacted.
     *
     * Append a newly created connection to the internal
     * m_connectionList list.
     */
    void connectionListAppend(QtTapioca::Connection * connection,
                              QtTapioca::Contact * contact);
    /**
     * @brief Slot called when a contact exists for a specific
     * request.
     * @param got Does the contact exists in the contact connector?
     * @param cookie The request cookie.
     *
     * This slot receives the response to a request from the
     * ContactConnector via ContactConnector::gotContact() and
     * passes it on to the ContactStateMachine matching the
     * request cookie.
     */
    void contactGot(bool got, quint64 cookie);
    /**
     * @brief Slot called when the URI's for a contact have
     * been got for a specific request.
     * @param uris A list of all the URIs found for the contact.
     * @param cookie The request cookie.
     *
     * This slot receives the response to a request from the
     * ContactConnector via ContactConnector::getURIs() and
     * passes it on to the ContactStateMachine matching the
     * request cookie.
     */
    void urisGot(QStringList uris, quint64 cookie);
    /**
     * @brief Slot called when a URI has been found for a
     * specific request.
     * @param contact_url The url of the contact matching the URI as
     *                    seen in the address book of the system.
     * @param cookie The request cookie.
     *
     * This slot receives the response to a request from the
     * ContactConnector via ContactConnector::findURI() and
     * passes it on to the ContactStateMachine matching the
     * request cookie.
     */
    void uriFound(const QString & contact_url, quint64 cookie);

    /**
     * @brief Slot called when a contact has been added for a
     * specific request.
     * @param contact_url The ID of the contact added.
     * @param cookie The request cookie.
     *
     * This slot receives the response to a request from the
     * ContactConnector via ContactConnector::addContact() and
     * passes it on to the ContactStateMachine matching the
     * request cookie.
     */
    void contactAdded(const QString & contact_url, quint64 cookie);
    /**
     * @brief Sends a response to the channel requester
     * indicating that an error occurred in setting up the
     * channel.
     * @param type The type of error to occur.
     * @param message The text of the error message.
     */
    void slotSendErrorReply(const QString & type, const QString & message);
    /**
     * @brief Carry out the actual work as documented for
     * ContactManager::contactContactUsingAccount().
     */
    quint64 contactContactUsingAccount(const QString & contact_url,
                                       const int account_handle,
                                       const int type,
                                       const QString & service,
                                       const QDBusObjectPath & path);
    /**
     * @brief Carry out the actual work as documented for
     * ContactManager::contactUrlUsingAccount().
     */
    quint64 contactUrlUsingAccount(const QString contact_uri,
                                   const int account_handle,
                                   const int type,
                                   const QString & service,
                                   const QDBusObjectPath & path);
    /**
     * @brief Carry out the actual work as documented for
     * ContactManager::contactUrl().
     */
    quint64 contactUrl(const QString contact_uri,
                       const int type,
                       const QString & service,
                       const QDBusObjectPath & path);

private:
    /**
     * @brief Create a new ContactStateMachine.
     * @param cookie The request cookie with which to create the ContactStateMachine.
     * @return A pointer to the new ContactStateMachine.
     *
     * This method creates a new ContactStateMachine object setting its cookie
     * to the one provided. It also makes appropriate signal/slot connections
     * between it and the ContactManagerPrivate instance creating it.
     */
    ContactStateMachine * createContactStateMachine(const quint64 cookie);

    QString getCMInternalUri(const QString & uri);

    QtTapioca::Contact * mapFromExternal(const QString & external_id);

    QString getProtocol(const QString & uri);

    /**
     * @brief A pointer to the non-private class instance.
     */
    ContactManager * const contactManager;
    /** @brief A pointer to the ContactConnector used. */
    ContactConnectorBase * m_connector;
    /**
     * @brief A mapping from the Tapioca Contact structure to the external ID
     * of the contact.
    */
    QHash<QtTapioca::Contact *, QString> m_telepathy2external;
    /**
     * @brief A mapping from the external ID of a contact to its Tapioca contact
     * structure.
     */
    QHash<QString, QtTapioca::Contact *> m_external2telepathy;
    /**
     * @brief A mapping between Connections and their Contacts.
     */
    QHash<const QtTapioca::Connection *,
          QList<QtTapioca::Contact *> > m_connectionList;
    /**
     * @brief A mapping between reques cookies and the ContactStateMachine for
     * each request.
     */
    QMap<quint64, ContactStateMachine*> m_stateMachines;

    QMap<quint64, QPair<QtTapioca::Contact *, QVariantMap> > m_pendingRegisterContacts;
};

#endif // header guard
