/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH <info@basyskom.de>
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *  @author George Goldberg <grundleborg@googlemail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "contactmanager_p.h"

#include "accountmanager.h"
#include "contactmanager.h"
#include "contactmanageradaptor.h"
#include "contactconnectorbase.h"
#include "contactstatemachine.h"
#include "cookiegenerator.h"

#include <QtCore/QDebug>
#include <QtCore/QTimer>

#include <QtTapioca/Connection>
#include <QtTapioca/Contact>
#include <QtTapioca/ContactList>
#include <QtTapioca/Handle>

#include <Decibel/Errors>

ContactManagerPrivate::ContactManagerPrivate(AccountManager * account_mgr,
                        ContactConnectorBase * connector,
                        ContactManager * contact_mgr) :
    accountManager(account_mgr),
    contactManager(contact_mgr),
    m_connector(connector)
{
    Q_ASSERT(0 != accountManager);
    Q_ASSERT(0 != m_connector);

    /*
     * connect up contactConnector signals
     * to local slots.
     */
    connect(m_connector, SIGNAL(contactGot(bool, quint64)),
            this, SLOT(contactGot(bool, quint64)));
    connect(m_connector, SIGNAL(urisGot(QStringList, quint64)),
            this, SLOT(urisGot(QStringList, quint64)));
    connect(m_connector, SIGNAL(uriFound(QString, quint64)),
            this, SLOT(uriFound(QString, quint64)));
    connect(m_connector, SIGNAL(contactAdded(QString, quint64)),
            this, SLOT(contactAdded(QString, quint64)));
}

ContactManagerPrivate::~ContactManagerPrivate()
{
    const QtTapioca::Connection * connection;
    foreach (connection, m_connectionList.keys())
    { deregisterContacts(connection); }
    //destruct all remaining state-machines
    const ContactStateMachine * csm;
    foreach (csm, m_stateMachines)
    {
        delete csm;
    }
    m_stateMachines.clear();

}

void
ContactManagerPrivate::deregisterContacts(const QtTapioca::Connection * connection)
{
    Q_ASSERT(0 != connection);
    if (!m_connectionList.contains(connection)) { return; }

    QtTapioca::Contact * internal_contact = 0;
    foreach (internal_contact, m_connectionList[connection])
    {
        if (!m_telepathy2external.contains(internal_contact)) { continue; }
        Q_ASSERT(m_telepathy2external.contains(internal_contact));
        QString external_url(m_telepathy2external[internal_contact]);
        Q_ASSERT(m_external2telepathy.contains(external_url));
        Q_ASSERT(m_external2telepathy[external_url] == internal_contact);

        m_external2telepathy.remove(external_url);
        m_telepathy2external.remove(internal_contact);
        Q_ASSERT(m_telepathy2external.size() == m_external2telepathy.size());
    }
    m_connectionList.remove(connection);
}

void
ContactManagerPrivate::setPresence(QtTapioca::Contact * internal_contact,
                                   const QtTapioca::PresenceState & presence,
                                   const QVariantMap & params)
{
    Q_ASSERT(0 != internal_contact);
    //Q_ASSERT(!presence.isEmpty());

    QString external_id(mapToExternal(internal_contact));
    if (0 == external_id) { return; }
    m_connector->setPresence(external_id, presence.name(), params, 0);
    // FIXME: is it OK that this is currently fire and forget?
}

QString
ContactManagerPrivate::mapToExternal(QtTapioca::Contact * contact)
{
    if (!m_telepathy2external.contains(contact)) { return 0; }
    else
    {
        Q_ASSERT(m_telepathy2external[contact] != 0);
        return m_telepathy2external[contact];
    }
}

void
ContactManagerPrivate::contactContactViaConnection(QtTapioca::Connection * const connection,
                            QtTapioca::Channel * const channel,
                            const quint64 cookie)
{
    // decouple execution of channel handlers:
    newChannelList.append(NewChannelInfo(connection, channel, cookie));
    QTimer::singleShot(10, contactManager, SLOT(onNewChannelCreated()));
}

void
ContactManagerPrivate::connectionListAppend(QtTapioca::Connection * connection, QtTapioca::Contact * contact)
{
    m_connectionList[connection].append(contact);
}

/**
 * @brief Register a new contact found on a connection.
 * @param protocol The protocol of the connection.
 * @param contact A pointer to the contact.
 * @param connection A pointer to the connection.
 *
 * Register a contact found on one of the connections in the mappings
 * to/from the external ID.
 */
void ContactManagerPrivate::registerContact(const QString & protocol,
                        QtTapioca::Contact * contact,
                        const QtTapioca::Connection * connection)
{
    Q_ASSERT(0 != connection);
    Q_ASSERT(0 != contact);
    Q_ASSERT(!protocol.isEmpty());

    QString uri(contact->uri());
    QString uri_protocol(getProtocol(uri));
    if (uri_protocol.isEmpty())
    { uri = protocol + QString("://") + uri; }
    else
    {
        if (protocol != uri_protocol)
        {
            qWarning() << "CtM: Fixing protocol in URI, was" << protocol
                        << "changed to" << uri_protocol;
            uri = uri_protocol + "://" + getCMInternalUri(uri);
        }
    }
    Q_ASSERT(uri.indexOf(QString("://") != 0));
    Q_ASSERT(getProtocol(uri) == protocol);
    Q_ASSERT(getCMInternalUri(uri) == getCMInternalUri(contact->uri()));

    // mark the handle as belonging to the given connection:
    m_connectionList[connection].append(contact);
    qDebug() << "Registering contact...";
    // Save contact data in a member var.
    QVariantMap contact_data;
    contact_data["URI"] = uri;
    contact_data["Name"] = contact->alias();
    quint64 cookie = CookieGenerator::instance()->getCookie();
    m_pendingRegisterContacts.insert(cookie, QPair<QtTapioca::Contact *, QVariantMap>(contact, contact_data));

    m_connector->findURI(uri, cookie);

}

QString
ContactManagerPrivate::getProtocol(const QString & uri)
{
    int pos = uri.indexOf(QString("://"));
    if (pos < 0) { return QString(); }
    else { return uri.left(pos); }
}

void ContactManagerPrivate::completeRegisterContact(const QString & contact_url,
                                                    const quint64 cookie)
{
    qDebug() << "Completeing contact registration.";
    if (contact_url.isEmpty())
    {
        // FIXME: Add data to PIM?
        qDebug() << "ContactManagerPrivate::registerContact(): contact not already in contactConnector, saving it there.";
        m_connector->addContact(m_pendingRegisterContacts.value(cookie).second, CookieGenerator::instance()->getCookie());
        m_pendingRegisterContacts.remove(cookie);
    }
    else
    {
        qDebug() << "ContactManagerPrivate::registerContact(): contact already in contactConnector. no need to resave. Should we update it?";
        Q_ASSERT(!contact_url.isEmpty());
        // FIXME: Do I need to update presence, etc. info here?
        m_telepathy2external.insert(m_pendingRegisterContacts.value(cookie).first, contact_url);
        m_external2telepathy.insert(contact_url, m_pendingRegisterContacts.value(cookie).first);
        m_pendingRegisterContacts.remove(cookie);
    }
    Q_ASSERT(m_telepathy2external.size() == m_external2telepathy.size());
}

void
ContactManagerPrivate::contactGot(bool got, quint64 cookie)
{
    /*
     * pass on signal by calling
     * the appropriate state machine's
     * method.
     */
    ContactStateMachine *csm = m_stateMachines.value(cookie);
    if(csm != 0)
    {
        csm->gotContact(got);
    }
}

void
ContactManagerPrivate::urisGot(QStringList uris, quint64 cookie)
{
    /*
     * pass on signal by calling
     * the appropriate state machine's
     * method.
     */
    ContactStateMachine *csm = m_stateMachines.value(cookie);
    if(csm != 0)
    {
        csm->gotUris(uris);
    }
}

void
ContactManagerPrivate::uriFound(const QString & contact_url, quint64 cookie)
{
    /*
     * pass on signal by calling
     * the appropriate state machine's
     * method.
     */
    ContactStateMachine *csm = m_stateMachines.value(cookie);
    if(csm != 0)
    {
        csm->uriFound(contact_url);
        return;
    }
    if(m_pendingRegisterContacts.value(cookie).first != 0)
    {
        completeRegisterContact(contact_url, cookie);
        return;
    }
}

void
ContactManagerPrivate::contactAdded(const QString & contact_url, quint64 cookie)
{
    /*
     * pass on signal by calling
     * the appropriate state machine's
     * method.
     */
    ContactStateMachine *csm = m_stateMachines.value(cookie);
    if(csm != 0)
    {
        csm->contactAdded(contact_url);
    }
}

void
ContactManagerPrivate::slotSendErrorReply(const QString & type, const QString & message)
{
    Q_UNUSED(type);     //Stop the compiler complaining
    Q_UNUSED(message);  //Stop the compiler complaining
    /*
     * first we send the error reply through DBus.
     */
    //contactManager->sendErrorReply(type, message);
    // FIXME: need a way to send an error reply. This method doesn't work in this context.
    qDebug() << "ContactManagerPrivate::slotSendErrorReply(): Error Occurred."
             << "    " << type << message;

    //remove ContactStateMachine from the list
    quint64 key = m_stateMachines.key(qobject_cast<ContactStateMachine *>(sender()));
    m_stateMachines.remove(key);

    /*
     * now we delete the connection state machine
     * since it is in error state.
     */
    sender()->deleteLater();
}

ContactStateMachine *
ContactManagerPrivate::createContactStateMachine(const quint64 cookie)
{
    /*
     * Create a new ContactStateMachine for
     * this request.
     */
    ContactStateMachine *csm = new ContactStateMachine(cookie,
                                                       m_connector,
                                                       accountManager,
                                                       contactManager,
                                                       &m_external2telepathy,
                                                       &m_telepathy2external);

    /*
     * connect up the state machine's signals
     * to slots here.
     */
    connect(csm,
            SIGNAL(sendErrorReply(const QString&, const QString&)),
            this,
            SLOT(slotSendErrorReply(const QString&, const QString&)));
    connect(csm, SIGNAL(contactConnected(QtTapioca::Connection * const, QtTapioca::Channel * const, const quint64)),
            this, SLOT(contactContactViaConnection(QtTapioca::Connection * const, QtTapioca::Channel * const, const quint64)));
    connect(csm, SIGNAL(connectionListAppend(QtTapioca::Connection *, QtTapioca::Contact *)),
            this, SLOT(connectionListAppend(QtTapioca::Connection *, QtTapioca::Contact *)));
    connect(csm,
            SIGNAL(registerRequestChannelHandler(const QString &,
                                                 const QDBusObjectPath &,
                                                 const quint64 &)),
            contactManager,
            SIGNAL(registerRequestChannelHandler(const QString &,
                                                 const QDBusObjectPath &,
                                                 const quint64 &)));

    /*
     * Insert this state machine into
     * the list of state machines, using
     * its cookie as the key.
     */
    m_stateMachines.insert(cookie, csm);

    //return a pointer to the new ContactStateMachine
    return csm;
}

quint64
ContactManagerPrivate::contactContactUsingAccount(const QString & contact_url,
                                                  const int account_handle,
                                                  const int type,
                                                  const QString & service,
                                                  const QDBusObjectPath & path)
{
    /*
     * Create the cookie for this request.
     */
    quint64 cookie = CookieGenerator::instance()->getCookie();

    /*
     * create a new ContactStateMachine
     */
    ContactStateMachine * csm = createContactStateMachine(cookie);

    /*
     * call the desired method on the state
     * machine.
     */
    csm->contactContactUsingAccount(contact_url, account_handle, type, service, path);

    /*
     * Return the cookie.
     */
    return cookie;
}

quint64
ContactManagerPrivate::contactUrlUsingAccount(const QString contact_uri,
                                    const int account_handle,
                                    const int type,
                                    const QString & service,
                                    const QDBusObjectPath & path)
{
    /*
     * Create the cookie for this request.
     */
    quint64 cookie = CookieGenerator::instance()->getCookie();

    /*
     * create a new ContactStateMachine
     */
    ContactStateMachine * csm = createContactStateMachine(cookie);

    /*
     * call the desired method on the state
     * machine.
     */
    csm->contactUrlUsingAccount(contact_uri, account_handle, type, service, path);

    /*
     * Return the cookie.
     */
    return cookie;
}

quint64
ContactManagerPrivate::contactUrl(const QString contact_uri,
                                    const int type,
                                    const QString & service,
                                    const QDBusObjectPath & path)
{
    /*
     * Create the cookie for this request.
     */
    quint64 cookie = CookieGenerator::instance()->getCookie();

    /*
     * create a new ContactStateMachine
     */
    ContactStateMachine * csm = createContactStateMachine(cookie);

    /*
     * call the desired method on the state
     * machine.
     */
    csm->contactUrl(contact_uri, type, service, path);

    /*
     * Return the cookie.
     */
    return cookie;
}

QString
ContactManagerPrivate::getCMInternalUri(const QString & uri)
{
    int pos = uri.indexOf(QString("://"));
    if (pos < 0) { return uri; }
    else { return uri.mid(pos + 3); }
}

/**
 * @brief Map a external contact ID to the internal Contact data.
 * @param external_id The contact's ID in the external PIM system.
 * @return A pointer to the internal Contact structure. This pointer is 0
 * if the contact is not know to telepathy.
 *
 * Map a external contact ID to the internal Contact data.
 */
QtTapioca::Contact *
ContactManagerPrivate::mapFromExternal(const QString & external_url)
{
    if (!m_external2telepathy.contains(external_url)) { return 0; }

    Q_ASSERT(m_external2telepathy.value(external_url) != 0);
    return m_external2telepathy.value(external_url);
}

