/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_DAEMON_CONTACTMANAGER_H
#define DECIBEL_DAEMON_CONTACTMANAGER_H

#include <QtCore/QObject>

#include <QtDBus/QDBusContext>

#include <Decibel/Types>

#include <QtTapioca/ContactBase>

namespace QtTapioca
{
class Connection;
}

class AccountManager;
class ContactManagerPrivate;
class ContactConnectorBase;

/**
 * @brief The ContactManager is a class interfacing to the desktop environment's
 *        contact data.
 *
 * The ContactManager is used by Decibel to query and update contact information
 * from the desktop environment's contact database. It does not expose an
 * interface to the contact database itself but makes it possible for an
 * application developer to interface with the ContactManager using the IDs
 * defined in the desktop environment's PIM database.
 *
 * The ContactManager leaves all the signaling of changes to the desktop
 * environment's PIM database.
 */
class ContactManager : public QObject, protected QDBusContext
{
    Q_OBJECT

public:
    /**
     * @brief Constructor
     * @param account_mgr Pointer to the AccountManager.
     * @param connector Pointer to the ContactConnector.
     * @param parent Parent Object.
     */
    explicit ContactManager(AccountManager * account_mgr,
                            ContactConnectorBase * connector,
                            QObject * parent = 0);
    /** @brief Destructor */
    ~ContactManager();

Q_SIGNALS:
    /**
     * @brief Emitted whenever a new channel is opened.
     *
     * This is necessary as Tapioca will only send signals for channels that
     * are created by outside entities.
     *
     * @param connection A pointer to the connection the channel belongs to.
     * @param channel A pointer to the channel.
     * @param cookie The request cookie
     */
    void channelOpened(QtTapioca::Connection * connection,
                       QtTapioca::Channel * channel, const quint64 cookie);

    /**
     * @brief Emitted whenever a new ChannelHandler needs to be registered.
     *
     * This signal is emitted whenever a ChannelHandler needs to be set up
     * to handle a request.
     *
     * @param service The service name of the new ChannelHandler.
     * @param path The object path of the new ChannelHandler.
     * @param request_cookie The request Id the new ChannelHandler should handle.
     */
    void registerRequestChannelHandler(const QString & service,
                                       const QDBusObjectPath & path,
                                       const quint64 & request_cookie);

public Q_SLOTS:
    /**
     * @brief Connect to a contact using a given account.
     * @param contact_url The url of the contact as seen in the addressbook of the
     *                    current environment.
     * @param account_handle A handle to the account to use.
     * @param type The type of connection requested.
     * @param service The service name that is to handle the newly created
     *                channel.
     * @param path The path to the object that is to handle the newly created
     *              channel.
     * @returns A request cookie.
     *
     * This is a basic connection method: It sets up a communication channel
     * between the account and the contact if possible and returns it to the
     * caller.
     *
     * If the service and/or path arguments are empty then the newly created
     * channel will get dispatched to a channel handler according to the
     * rules defined in the policy engine. Otherwise the channel will be
     * forwarded to the defined channel handler directly.
     */
    quint64
    contactContactUsingAccount(const QString & contact_url,
                               const int account_handle,
                               const int type, const QString & service,
                               const QDBusObjectPath & path);

    /**
     * @brief Connect to a somebody using a given url.
     * @param contact_url The URL of the contact to connect to.
     * @param account_handle A handle to the account to use.
     * @param type The type of connection requested.
     * @param service The service name that is to handle the newly created
     *                channel.
     * @param path The path to the object that is to handle the newly created
     *              channel.
     * @returns A request cookie.
     *
     * This is a basic connection method: It sets up a communication channel
     * between the account and a url if possible and returns this channel to the
     * caller.
     */
    quint64
    contactUrlUsingAccount(const QString & contact_url, const int account_handle,
                           const int type, const QString & service,
                           const QDBusObjectPath & path);

    /**
     * @brief Connect to a contact URL using any account available.
     * @param contact_url The URL of the contact to connect to.
     * @param type The type of connection requested.
     * @param service The service name that is to handle the newly created
     *                channel.
     * @param path The path to the object that is to handle the newly created
     *              channel.
     * @returns A request cookie.
     *
     * This connection method lets Decibel decide which of the accounts to use
     * to connect to the given URL. It sets up a communication channel between
     * an account it selected and the url if possible and returns this channel
     * to the caller.
     */
    quint64
    contactUrl(const QString & contact_url, const int type,
               const QString & service, const QDBusObjectPath & path);


private Q_SLOTS:
    void onConnectionOpened(QtTapioca::Connection *);
    void onConnectionClosed(QtTapioca::Connection *);
    void onPresenceUpdated(QtTapioca::ContactBase *, const QtTapioca::PresenceState &,
                           const QVariantMap &);

    void onNewChannelCreated();

private:
    ContactManagerPrivate * const d;

    friend class ContactManagerPrivate;
};

#endif
