/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH <info@basyskom.de>
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "contactmanager.h"
#include "contactmanager_p.h"
#include "contactmanageradaptor.h"

#include "accountmanager.h"
#include "contactconnectorbase.h"
#include "contactstatemachine.h"

#include <Decibel/Errors>

#include <QtCore/QHash>
#include <QtCore/QCoreApplication>
#include <QtCore/QPointer>
#include <QtCore/QDebug>
#include <QtCore/QTimer>

#include <QtTapioca/Connection>
#include <QtTapioca/Contact>
#include <QtTapioca/ContactList>
#include <QtTapioca/Handle>

ContactManager::ContactManager(AccountManager * account_mgr,
                               ContactConnectorBase * connector,
                               QObject * parent) :
        QObject(parent),
        d(new ContactManagerPrivate(account_mgr, connector, this))
{
    Q_ASSERT(0 != d);
    d->adaptor = new ContactManagerAdaptor(this);
    Q_ASSERT(0 != d->adaptor);
}

ContactManager::~ContactManager()
{ delete d; }

quint64
ContactManager::contactUrlUsingAccount(const QString & contact_uri,
                                       const int account_handle,
                                       const int type,
                                       const QString & service,
                                       const QDBusObjectPath & path)
{
   return d->contactUrlUsingAccount(contact_uri, account_handle, type, service, path);
}

quint64
ContactManager::contactContactUsingAccount(const QString & contact_url,
                                           const int account_handle,
                                           const int type,
                                           const QString & service,
                                           const QDBusObjectPath & path)
{
    qDebug() << "ContactManager::contactContactUsingAccount():" << "Called.";
    return d->contactContactUsingAccount(contact_url, account_handle, type, service, path);
}

quint64
ContactManager::contactUrl(const QString & contact_uri,
                           const int type,
                           const QString & service,
                           const QDBusObjectPath & path)
{
    return d->contactUrl(contact_uri, type, service, path);
}

void ContactManager::onConnectionOpened(QtTapioca::Connection * connection)
{
    Q_ASSERT(0 != connection);
    Q_ASSERT(connection->status() == QtTapioca::Connection::Connected);

    const QtTapioca::ContactList * contact_list = connection->contactList();
    Q_ASSERT(0 != contact_list);

    const QString protocol(connection->protocol());
    Q_ASSERT(!protocol.isEmpty());

    QList<QtTapioca::Contact *> known_contacts = contact_list->knownContacts();
    QtTapioca::Contact * internal_contact = 0;
    foreach (internal_contact, known_contacts)
    {
        Q_ASSERT(0 != internal_contact);

        QtTapioca::Handle * internal_handle = internal_contact->handle();
        Q_ASSERT(0 != internal_handle);
        // FIXME: Handle non-contact handles properly:
        switch (internal_handle->type())
        {
            case QtTapioca::Handle::Contact:
                d->registerContact(protocol, internal_contact, connection);
                connect(internal_contact,
                        SIGNAL(presenceUpdated(QtTapioca::ContactBase *,
                                               const QtTapioca::PresenceState &,
                                               const QVariantMap &)),
                        this,
                        SLOT(onPresenceUpdated(QtTapioca::ContactBase *,
                                               const QtTapioca::PresenceState &,
                                               const QVariantMap &)));
                break;
            case QtTapioca::Handle::Room:
            case QtTapioca::Handle::List:
            case QtTapioca::Handle::None:
                // FIXME: Handle other types of handles...
                break;
            default:
                qWarning() << "Unknown Handle type encountered.";
        }
    }
}

void ContactManager::onConnectionClosed(QtTapioca::Connection * connection)
{
    Q_ASSERT(0 != connection);
    // This can hit all connections due to a crash, etc. so it does not
    // make sense to assert the status of the connection here.

    d->deregisterContacts(connection);
}

void ContactManager::onPresenceUpdated(QtTapioca::ContactBase * internal_contactbase,
                                       const QtTapioca::PresenceState & presence,
                                       const QVariantMap & params)
{
    Q_ASSERT(0 != internal_contactbase);
    QtTapioca::Contact * internal_contact =
        dynamic_cast<QtTapioca::Contact *>(internal_contactbase);
    if (0 == internal_contact) { return; }

    const QtTapioca::Handle * handle = internal_contact->handle();
    Q_ASSERT(0 != handle);
    switch(handle->type())
    {
        case QtTapioca::Handle::Contact:
            d->setPresence(internal_contact, presence, params);
            break;
        case QtTapioca::Handle::Room:
        case QtTapioca::Handle::List:
        case QtTapioca::Handle::None:
            // FIXME: Handle other types of handles...
            break;
        default:
            qWarning() << "Unknown Handle type encountered.";
    }
}

void ContactManager::onNewChannelCreated()
{
    ContactManagerPrivate::NewChannelInfo info;
    foreach (info, d->newChannelList)
    { emit channelOpened(info.connection, info.channel, info.cookie); }
    d->newChannelList.clear();
}
