/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "connectionfacade.h"
#include "protocolmanager.h"

#include <QtCore/QCoreApplication>
#include <QtCore/QDebug>
#include <QtCore/QPointer>

#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusConnectionInterface>

#include <QtTapioca/ConnectionManager>
#include <QtTapioca/ConnectionManagerFactory>
#include <QtTapioca/PresenceState>
#include <QtTapioca/UserContact>

#include <Decibel/Errors>
#include <Decibel/AccountData>

/// @cond INCLUDE_PRIVATE

/**
 * @brief Private data class for the ConnectionFacade.
 * @author Tobias Hunger <info@basyskom.de>
 */
class ConnectionFacadePrivate
{
public:
    ConnectionFacadePrivate(ProtocolManager * const protocol_mgr) :
        protocolMgr(protocol_mgr),
        acceptConnections(true)
    { Q_ASSERT(0 != protocol_mgr); }

    /** @brief A pointer to the ProtocolManager. */
    ProtocolManager * const protocolMgr;

    /** @brief A list of known Connections. */
    QList<QtTapioca::Connection *> connections;

    bool acceptConnections;
};

/// @endcond

// -----------------------------------------------------------------------

ConnectionFacade::ConnectionFacade(ProtocolManager* protocol_mgr,
                                   QObject * parent) :
    QObject(parent),
    d(new ConnectionFacadePrivate(protocol_mgr))
{
    connect(QDBusConnection::sessionBus().interface(),
            SIGNAL(serviceOwnerChanged(const QString &, const QString &, const QString &)),
            this,
            SLOT(onServiceOwnerChanged(const QString &, const QString &, const QString &)));
}

ConnectionFacade::~ConnectionFacade()
{ delete d; }

void ConnectionFacade::onAboutToQuit()
{
    d->acceptConnections = false;

    emit shuttingDown();

    // Disconnect from all Connections:
    foreach (QtTapioca::Connection * connection, d->connections)
    { connection->disconnect(); }

    // Wait for disconnection events:
    QCoreApplication * const qapp(QCoreApplication::instance());
    while (!d->connections.isEmpty())
    { qapp->processEvents(); }
}

QtTapioca::Connection *
ConnectionFacade::connectUsingAccountAndConnectionManager(const QVariantMap& account_data,
                                                          const QString & cm_name,
                                                          const QtTapioca::PresenceState & presence_state,
                                                          const QVariantMap & params)
{
    Q_ASSERT(!account_data.isEmpty());
    Q_ASSERT(account_data.contains(Decibel::name_protocol));

    if (!d->acceptConnections)
    { return 0; }

    QString protocol(account_data[Decibel::name_protocol].toString());
    // FIXME: take advantage of the PresenceState object - don't just check against the name!
    Q_ASSERT(!Decibel::isOffline(presence_state.name()));
    // AccountManager should not have called us if the account is
    // supposed to be offline!

    // Get CM:
    QtTapioca::ConnectionManager * current_cm =
            QtTapioca::ConnectionManagerFactory::self()->
            getConnectionManagerByName(cm_name);

    if (current_cm == 0) { return 0; }

    Q_ASSERT(current_cm != 0);

    // Map parameters:
    const QList<QtTapioca::ConnectionManager::Parameter>
            cm_params(current_cm->protocolParameters(protocol));
    QList<QtTapioca::ConnectionManager::Parameter> account_params;
    QString key;
    foreach (const QtTapioca::ConnectionManager::Parameter & current_param, cm_params)
    {
        key = current_param.name();
        if (account_data.contains(key) &&
            account_data[key].typeName() ==
                current_param.value().typeName())
        {
            account_params.append(QtTapioca::ConnectionManager::Parameter(key,
                                                                          account_data[key]));
        }
        else if (current_param.flags() & QtTapioca::ConnectionManager::Parameter::Required)
        { return 0; }
    }

    // Create connection:
    QtTapioca::Connection* connection = current_cm->requestConnection(protocol, account_params);
    if (connection == 0)
    { return 0; }

    Q_ASSERT(connection != 0);

    Q_ASSERT(!d->connections.contains(connection));
    d->connections.append(connection);

    // register connection:
    connect(connection,
            SIGNAL(statusChanged(QtTapioca::Connection*,
                                 QtTapioca::Connection::Status,
                                 QtTapioca::Connection::Reason)),
            this,
            SLOT(onStatusChanged(QtTapioca::Connection *,
                                 QtTapioca::Connection::Status,
                                 QtTapioca::Connection::Reason)));
    connect(connection,
            SIGNAL(channelCreated(QtTapioca::Connection*,
                                  QtTapioca::Channel*, bool)),
            this,
            SLOT(onChannelOpened(QtTapioca::Connection*,
                                 QtTapioca::Channel*, bool)));
    // Is there a disconnected() signal or something we can connect to
    // in order to know when the connection goes down?
    // NO NEED! Its looked after by statusChanged().

    // do the actual connect:
    connection->connect(presence_state, params);

    return connection;
}

QtTapioca::Connection *
ConnectionFacade::connectUsingAccount(const QVariantMap & account_data,
                                      const QtTapioca::PresenceState & presence_state,
                                      const QVariantMap & params)
{
    qDebug() << "ConnectionFacade::connectUsingAccount():"
             << "Connecting using given account...";
    QString protocol = account_data[Decibel::name_protocol].toString();
    // Get a connection manager to use:
    QString cm_name(d->protocolMgr->defaultConnectionManagerForProtocol(protocol));

    if (cm_name.isEmpty()) { return 0; }

    return connectUsingAccountAndConnectionManager(account_data, cm_name,
                                                   presence_state, params);
}

void ConnectionFacade::onStatusChanged(QtTapioca::Connection* connection,
                                       QtTapioca::Connection::Status new_state,
                                       QtTapioca::Connection::Reason reason)
{
    // find connection_handle:
    if (0 == connection) { return; }
    qDebug() << "ConnectionFacade::onStatusChanged():"
             << "Connection" << connection << "status changed:\n"
             << "    new state:" << new_state << "\n"
             << "    reason:" <<  reason;

    // This is one of our connections:
    Q_ASSERT(0 != connection); // 0 connection may not end up in d->connections!

    if (new_state == QtTapioca::Connection::Connected)
    {
        Q_ASSERT(0 != connection->userContact());
        qDebug() << "ConnectionFacade::onStatusChanged():"
                 << "Status changed to CONNECTED. Adding connection to the list.";
        addConnection(connection);
        return;
    }
    else if (new_state == QtTapioca::Connection::Disconnected)
    {
        Q_ASSERT(connection->status() == QtTapioca::Connection::Disconnected);
        qDebug() << "ConnectionFacade::onStatusChanged():"
                 << "Status changed to DISCONNECTED. Removing connection from the list.";
        rmConnection(connection);
        return;
    }
    qDebug() << "ConnectionFacade::onStatusChanged():"
             << "Status changed, but it wasn't to CONNECTED or DISCONNECTED. Doing nothing.";
}

void ConnectionFacade::onPresenceUpdated(QtTapioca::ContactBase * contact_info,
                                         const QtTapioca::PresenceState & presence_state,
                                         const QVariantMap & params)
{
    Q_ASSERT(contact_info != 0);
    qDebug() << "presence changed:" << contact_info->uri() << presence_state.name();
    const QtTapioca::UserContact* user_contact =
            dynamic_cast<const QtTapioca::UserContact*>(contact_info);
    if (user_contact == 0) { return; } // Not a user contact after all...

    // find the connection handle for this user_contact:
    foreach (QtTapioca::Connection * connection, d->connections)
    {
        Q_ASSERT(connection != 0);
        if (connection->userContact() == user_contact)
        {
            emit ownPresenceUpdated(connection, presence_state, params);
            return;
        }
    }
}

void ConnectionFacade::onChannelOpened(QtTapioca::Connection * connection,
                                       QtTapioca::Channel * channel,
                                       bool suppress_handler)
{
    qDebug() << "Got a new channel!";
    if (connection == 0 || channel == 0 || suppress_handler)
    { return; }

    // We must do something with this channel...
    Q_ASSERT(connection != 0);
    Q_ASSERT(channel != 0);
    Q_ASSERT(connection->status() == QtTapioca::Connection::Connected);
    emit channelOpened(connection, channel, 0);
}

QtTapioca::PresenceState ConnectionFacade::updatePresence(QtTapioca::Connection * connection,
                                         const QtTapioca::PresenceState & presence_state,
                                         const QVariantMap & params)
{
    if (0 == connection) { return QtTapioca::PresenceState(QString()); }

    Q_ASSERT(connection != 0);

    if (connection->userContact() == 0 ||
        !connection->userContact()->setPresenceWithParameters(presence_state, params)) // not connected!
    { return QtTapioca::PresenceState(Decibel::state_offline, QtTapioca::PresenceState::OfflineType); }

    return presence_state;
}

void ConnectionFacade::onServiceOwnerChanged(const QString & service,
                                             const QString & /* old_owner */,
                                             const QString & new_owner)
{
    if (!new_owner.isEmpty()) { return; }

    foreach (QtTapioca::Connection * connection, d->connections)
    {
        if (connection->serviceName() == service)
        { rmConnection(connection); }
    }
}

void ConnectionFacade::addConnection(QtTapioca::Connection * const connection)
{
    Q_ASSERT(0 != connection);
    Q_ASSERT(0 != connection->userContact());
    Q_ASSERT(connection->status() == QtTapioca::Connection::Connected);

    connect(connection->userContact(),
            SIGNAL(presenceUpdated(QtTapioca::ContactBase*,
                                   const QtTapioca::PresenceState &,
                                   const QVariantMap &)),
            SLOT(onPresenceUpdated(QtTapioca::ContactBase*,
                                   const QtTapioca::PresenceState &,
                                   const QVariantMap &)));

    emit connectionOpened(connection);

    // Fake presence information for the AccountManager:
    if (!connection->hasPresenceSupport())
    {
        emit ownPresenceUpdated(connection,
            QtTapioca::PresenceState(Decibel::state_available, QtTapioca::PresenceState::AvailableType),
                                QVariantMap());
    }

}

void ConnectionFacade::rmConnection(QtTapioca::Connection * const connection)
{
    Q_ASSERT(0 != connection);

    // Fake presence information for the AccountManager:
    if (!connection->hasPresenceSupport())
    {
        emit ownPresenceUpdated(connection, QtTapioca::PresenceState(Decibel::state_offline, QtTapioca::PresenceState::OfflineType),
                                QVariantMap());
    }

    emit connectionClosed(connection);

    int num_removals = d->connections.removeAll(connection);
    Q_ASSERT(num_removals == 1);

    delete connection;
}
