/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "accountmanager.h"
#include "accountconnectorbase.h"
#include "accountmanageradaptor.h"
#include "connectionfacade.h"

#include <QtTapioca/Connection>
#include <QtTapioca/ContactBase>

#include <Decibel/Errors>
#include <Decibel/AccountData>

#include <QtCore/QSettings>
#include <QtCore/QHash>
#include <QtCore/QPointer>
#include <QtCore/QDebug>

namespace
{
    static const QString accountmanager_group("AccountManager");
    static const QString account_array("Accounts");
}

/// @cond INCLUDE_PRIVATE

/**
 * @brief Private class implementing the AccountManager.
 *
 * A private class implementing the AccountManager.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */
class AccountManagerPrivate
{
public:
    /**
     * @brief Constructor.
     */
    AccountManagerPrivate(ConnectionFacade * conn_mgr, AccountConnectorBase * connector) :
        m_connectionFacade(conn_mgr),
        m_blockReconnects(false),
        m_connector(connector),
        m_data_is_available(false)
    {
        Q_ASSERT(0 != m_connectionFacade);
        Q_ASSERT(0 != m_connector);
    }

    /**
     * @brief Destructor.
     */
    ~AccountManagerPrivate()
    { }

// Helper functions:

    /**
     * @brief Create an account in the accounts database.
     * @param nv_pairs The account data.
     * @return A handle to the account.
     *
     * This method creates an account in the account database. It assumes
     * the account data is complete.
     */
    uint createAccount(const QVariantMap & nv_pairs)
    {
        Q_ASSERT(nv_pairs.contains(Decibel::name_display_name));
        Q_ASSERT(nv_pairs.contains(Decibel::name_protocol));
        Q_ASSERT(nv_pairs.contains(Decibel::name_presence));
        Q_ASSERT(nv_pairs.contains(Decibel::name_presence_parameters));
        Q_ASSERT(nv_pairs.contains(Decibel::name_current_presence));
        Q_ASSERT(nv_pairs.contains(Decibel::name_autoreconnect));

        qDebug() << "Storing account data:";
        foreach(const QString key, nv_pairs.keys())
        {
            qDebug() << "Key:" << key << "value:" << nv_pairs[key];
        }

        return m_connector->storeAccount(nv_pairs);
    }

    /**
     * @brief Make sure a account has the required presence information.
     * @param nv_pairs The account data.
     * @returns A new set of account data.
     *
     * Make sure a account has the required presence information and fill in
     * the blanks if there are any.
     */
    QVariantMap sanityCheck(const QVariantMap & nv_pairs) const
    {
        QVariantMap pairs(nv_pairs);

        if (!pairs.contains(Decibel::name_protocol))
        {
            return QVariantMap();
        }

        if (!pairs.contains(Decibel::name_presence) ||
            pairs[Decibel::name_presence].value<QtTapioca::PresenceState>().name().isEmpty())
        {
            QVariant presenceStateVariant;
            presenceStateVariant.setValue(QtTapioca::PresenceState(Decibel::state_offline, QtTapioca::PresenceState::OfflineType));
            pairs.insert(Decibel::name_presence,
                         presenceStateVariant);
        }
        if (!pairs.contains(Decibel::name_presence_parameters))
        { pairs.insert(Decibel::name_presence_parameters, QVariant(QVariantMap())); }
        if (!pairs.contains(Decibel::name_autoreconnect))
        { pairs.insert(Decibel::name_autoreconnect, QVariant(false)); }
        if (!pairs.contains(Decibel::name_display_name))
        {
            // create a sensible display name:
            QString protocol;
            QString account;
            QString server;
            QString port;

            if (pairs.contains(Decibel::name_protocol))
            { protocol = pairs[Decibel::name_protocol].toString(); }
            if (pairs.contains(Decibel::name_tp_account))
            { account = pairs[Decibel::name_tp_account].toString(); }
            if (pairs.contains(Decibel::name_tp_server))
            { server = pairs[Decibel::name_tp_server].toString(); }
            if (pairs.contains(Decibel::name_tp_port))
            { port = pairs[Decibel::name_tp_port].toString(); }

            if (protocol.isEmpty()) { protocol = "unknown"; }
            if (account.isEmpty()) { account = "unknown"; }
            if (server.isEmpty()) { server = "unknown"; }
            if (!port.isEmpty()) { port = ':' + port; }

            QString name(protocol + "://" +
                         account + '@' + server + port + '/');
            pairs.insert(Decibel::name_display_name, QVariant(name));
        }

        if (!pairs.contains(Decibel::name_current_presence) ||
            pairs[Decibel::name_current_presence].value<QtTapioca::PresenceState>().name().isEmpty())
        {
            QVariant currentPresenceStateVariant;
            currentPresenceStateVariant.setValue(
                    QtTapioca::PresenceState(
                            Decibel::state_offline,
                            QtTapioca::PresenceState::OfflineType));
            pairs.insert(Decibel::name_current_presence,
                         currentPresenceStateVariant);
        }

        Q_ASSERT(pairs.contains(Decibel::name_protocol));
        Q_ASSERT(pairs.contains(Decibel::name_presence));
        Q_ASSERT(pairs.contains(Decibel::name_current_presence));
        Q_ASSERT(pairs.contains(Decibel::name_autoreconnect));
        Q_ASSERT(pairs.contains(Decibel::name_presence_parameters));
        Q_ASSERT(pairs.contains(Decibel::name_display_name));
        Q_ASSERT(pairs.size() >= nv_pairs.size() &&
                 pairs.size() <= nv_pairs.size() + 5 );

        return pairs;
    }

    QVariantMap getAccount(const uint account_handle)
    { return sanityCheck(m_connector->getAccount(account_handle)); }

    /**
     * @brief Manage presence information on a connection.
     * @param account_handle A handle to an account.
     * @param presence_state The presence state.
     * @param message A presence message.
     * @return A value != "Offline" on success and a empty string otherwise.
     *
     * Change the presence information on an account. This brings an account
     * on/off line as required.
     */
    QString manageConnection(const uint account_handle,
                             const QtTapioca::PresenceState & presence_state,
                             const QVariantMap & params)
    {
        // Before calling this method, we should check that the AccountConnector
        // contains the requested account, and that the presence state requested
        // is valid. So, now, we can assert these assumptions.
        Q_ASSERT(m_connector->hasAccount(account_handle));

        QtTapioca::PresenceState result;

        QtTapioca::Connection * connection = connectionOf(account_handle);

        if (Decibel::isOffline(presence_state.name()))  // FIXME: Check the entire PresenceState object, not name
        {
            qDebug() << "AccountManagerPrivate::manageConnection():"
                     << "Going offline now with account" << account_handle;
            // We need to disconnect:
            if (0 != connection)
            { connection->disconnect(); }
            // connection pointer is set on signal!
            //
            // We do not delete the connection here since we still need it to
            // send signals about it disconnecting.
        }
        else
        {
            if (connection == 0)
            {
                qDebug() << "AccountManagerPrivate::manageConnection():"
                         << "Going online now with account" << account_handle;
                // We need to connect if we are not yet connected.
                connection =
                    m_connectionFacade->connectUsingAccount(getAccount(account_handle),
                                                            presence_state, params);
                if (connection == 0)
                {
                    // FIXME: Give proper error code!
                    return QString();
                }
                setConnection(account_handle, connection);
            }
            else
            {
                qDebug() << "AccountManagerPrivate::manageConnection():"
                         << "Staying online with account" << account_handle;
                // update presence info
                result =
                    m_connectionFacade->updatePresence(connection,
                                                       presence_state,
                                                       params);
            }
            Q_ASSERT(0 != connectionOf(account_handle));
        }
        return result.name();
    }

    /**
     * @brief Associate a account with a connection.
     * @param account_handle A handle to an account.
     * @param connection A pointer to an Connection.
     *
     * Associate an account with a connection.
     */
    void setConnection(const uint account_handle,
                       QtTapioca::Connection * connection)
    {
        Q_ASSERT(0 == connectionOf(account_handle) ||
                 0 == connection);
        if (!m_connector->hasAccount(account_handle)) { return; }
        m_connections[account_handle] = connection;
    }

    /**
     * @brief Get the connection an account is using.
     * @param account_handle A handle to an account.
     * @return A pointer to the connection the account is using. This
     *         connection will be 0 if the account is not on line.
     */
    QtTapioca::Connection * connectionOf(const uint account_handle) const
    {
        if (!m_connector->hasAccount(account_handle)) { return 0; }
        return m_connections[account_handle];
    }

    /**
     * @brief Find the account using a connection.
     * @param connection A pointer to an Connection.
     * @return A handle to an account. This handle is 0 if no account is using
     * that connection.
     *
     * Find the account using a connection.
     */
    uint findAccountWithConnection(QtTapioca::Connection * connection)
    {
        const QList<uint> values(m_connections.keys());
        foreach (uint account_handle, values)
        {
            if (m_connections[account_handle] == connection)
            { return account_handle; }
        }
        return 0;
    }

    /**
     * @brief Update presence information on an account.
     * @param account_handle A handle to an account.
     * @param presence_state The new presence state.
     * @param params The new presence parameters.
     *
     * Update presence information on an account, bringing it on/off line as
     * required.
     */
    void updatePresenceLocally(const uint account_handle,
                               const QtTapioca::PresenceState & presence_state,
                               const QVariantMap params)
    {
        QVariantMap account_data = getAccount(account_handle);
        Q_ASSERT(!account_data.isEmpty());

         QVariant presenceStateVariant;
         presenceStateVariant.setValue(presence_state);
        account_data[Decibel::name_current_presence] = presenceStateVariant;
        if (!params.isEmpty())
        { account_data[Decibel::name_presence_parameters] = QVariant(params); }

        account_data = sanityCheck(account_data);

        if (account_data.isEmpty()) { return; }

        m_connector->updateAccount(account_handle, account_data);

        // Get back to intended presence state:
        if (m_blockReconnects) { return; }

        if (account_data[Decibel::name_autoreconnect].toBool())
        {
            const QtTapioca::PresenceState intended_state(account_data[Decibel::name_presence].toString());    // FIXME: PresenceState
            // FIXME: Should we compare the actual presence state objects rather than their names?
            if (intended_state.name() != presence_state.name())
            {
                manageConnection(account_handle, intended_state,
                    account_data[Decibel::name_presence_parameters].toMap());
            }
        }
    }

    /**
     * @brief Update the presence message on an account.
     * @param account_handle A handle to an account.
     * @param params The presence parameters set.
     *
     * Update the presence message on an account.
     */
    void updatePresenceMessageLocally(const uint account_handle,
                                      const QVariantMap & params)
    {
        QVariantMap account_data = getAccount(account_handle);
        account_data[Decibel::name_presence_parameters] = QVariant(params);
        m_connector->updateAccount(account_handle, account_data);
    }

    /** @brief Account data, addressed by account_handle. */
    QHash<uint, QtTapioca::Connection *> m_connections;

    /** @brief A pointer to the ConnectionFacade */
    QPointer<ConnectionFacade> m_connectionFacade;
    /** @brief A pointer to the D-Bus Adaptor of the AccountManager. */
    QPointer<AccountManagerAdaptor> m_adaptor;

    bool m_blockReconnects;

    AccountConnectorBase * const m_connector;

    bool m_data_is_available;

    friend class AccountManager;
};

/// @endcond

// ****************************************************************************

AccountManager::AccountManager(ConnectionFacade * connection_facade,
                               AccountConnectorBase * connector,
                               QObject * parent) :
    QObject(parent),
    d(new AccountManagerPrivate(connection_facade, connector))
{
    Q_ASSERT(d != 0);
    d->m_adaptor = new AccountManagerAdaptor(this);
    Q_ASSERT(d->m_adaptor != 0);

    connect(connection_facade,
        SIGNAL(ownPresenceUpdated(QtTapioca::Connection *,QtTapioca::PresenceState,QVariantMap)),
        SLOT(onOwnPresenceUpdated(QtTapioca::Connection *,QtTapioca::PresenceState,QVariantMap)));
    connect(connection_facade,
        SIGNAL(connectionClosed(QtTapioca::Connection *)),
        SLOT(onConnectionClosed(QtTapioca::Connection *)));
    connect(d->m_connector, SIGNAL(accountDataAvailable(bool)),
            SLOT(doAccountDataAvailable(bool)));
}

AccountManager::~AccountManager()
{ delete d; }

void AccountManager::bringUpAccounts()
{
    const QList<uint> account_ids(d->m_connector->accountIds());
    foreach (uint id, account_ids)
    {
        QVariantMap account_data = d->getAccount(id);
        Q_ASSERT(!account_data.isEmpty());

        qDebug() << "Bringing up" << id << ":"
                 << account_data[Decibel::name_protocol]
                 << "intended presence:"
                 << account_data[Decibel::name_presence]
                 << "current presence:"
                 << account_data[Decibel::name_current_presence]
                 << "presence parameters:"
                 << account_data[Decibel::name_presence_parameters];

        if (Decibel::isOffline(account_data[Decibel::name_presence].toString()))    // FIXME: PresenceState
        { continue; }

        setPresenceAndParameters(id,
                                 QtTapioca::PresenceState(account_data[Decibel::name_presence].toString()), // FIXME: PresenceState
                                 account_data[Decibel::name_presence_parameters].toMap());
    }
}

void AccountManager::bringDownAccounts()
{
    const QList<QtTapioca::Connection *> values(d->m_connections.values());
    foreach (QtTapioca::Connection * conn, values)
    { conn->disconnect(); }
}


void AccountManager::doAccountDataAvailable(const bool is_available)
{
    d->m_data_is_available = is_available;
    d->m_blockReconnects = !(is_available);

    emit resetData();

    if (is_available) { bringUpAccounts(); }
    else { bringDownAccounts(); }
}

QString AccountManager::protocol(const uint account_handle) const
{ return d->m_connector->protocol(account_handle); }

bool AccountManager::gotAccount(const uint account_handle) const
{ return d->m_connector->hasAccount(account_handle); }

QtTapioca::Connection *
AccountManager::connectionOf(const uint account_handle) const
{ return d->connectionOf(account_handle); }

QList<uint> AccountManager::listAccounts() const
{ return d->m_connector->accountIds(); }

QVariantMap AccountManager::queryAccount(const uint account_handle) const
{
    if (!gotAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
    }
    return d->getAccount(account_handle);
}

uint AccountManager::addAccount(const QVariantMap & nv_pairs)
{
    QList<uint> matching_accounts = findAccounts(nv_pairs);
    if (!matching_accounts.isEmpty())
    {
        Q_ASSERT(matching_accounts.size() == 1);
        return matching_accounts[0];
    }

    QVariantMap data = d->sanityCheck(nv_pairs);
    if (data.isEmpty())
    {
        sendErrorReply(Decibel::ErrorDataIncomplete,
                       tr("Account data was incomplete"));
        return 0;
    }
    uint account_handle = d->createAccount(data);

    if (account_handle != 0) { emit accountCreated(account_handle); }
    else
    {
        sendErrorReply(Decibel::ErrorInternalError,
                       tr("Failed to store account information."));
        return 0;
    }

    return account_handle;
}

void AccountManager::updateAccount(const uint account_handle,
                                   const QVariantMap & nv_pairs)
{
    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return;
    }

    QVariantMap data(nv_pairs);
    // Remove internal stuff:
    data.remove(Decibel::name_current_presence);
    QtTapioca::PresenceState presence_state(data[Decibel::name_presence].value<QtTapioca::PresenceState>());
    QVariantMap presence_params = data[Decibel::name_presence_parameters].toMap();
    data.remove(Decibel::name_presence);
    data.remove(Decibel::name_presence_parameters);

    // merge old and new data:
    QVariantMap old_data = queryAccount(account_handle);
    const QStringList values(old_data.keys());
    foreach (const QString & key, values)
    {
        if (!data.contains(key)) { data.insert(key, old_data[key]); }
    }

    // make sure the data is sane:
    data = d->sanityCheck(data);

    if (data.isEmpty())
    {
        sendErrorReply(Decibel::ErrorDataIncomplete,
                       tr("Update invalidates the data."));
        return;
    }

    QtTapioca::PresenceState old_presence_state(old_data[Decibel::name_presence].value<QtTapioca::PresenceState>());
    QVariantMap old_presence_params(old_data[Decibel::name_presence_parameters].toMap());

    d->m_connector->updateAccount(account_handle, data);

    if ((presence_state == old_presence_state) &&
        (presence_params != old_presence_params))
    { setPresenceAndParameters(account_handle, presence_state, presence_params); }
    else if (presence_state != old_presence_state)
    { setPresence(account_handle, presence_state); }
    else if (presence_params != old_presence_params)
    { setPresenceParameters(account_handle, presence_params); }

    emit accountUpdated(account_handle);
}

void AccountManager::deleteAccount(const uint account_handle)
{
    if (d->m_connector->deleteAccount(account_handle))
    { emit accountDeleted(account_handle); }
}

QList<uint> AccountManager::findAccounts(const QVariantMap & nv_pairs) const
{ return d->m_connector->findAccounts(nv_pairs); }

QtTapioca::PresenceState AccountManager::setPresence(const uint account_handle,
                                    const QtTapioca::PresenceState & presence_state)
{
    qDebug() << "Req: setPresence: handle="<< account_handle
             << "state=" << presence_state.name();

    // Check the presence state requested for the account is valid. If it isn't,
    // send an error reply and return.
    if (presence_state.name().isEmpty())    // FIXME: is checking the name all that is necessary?
    {
        sendErrorReply(Decibel::ErrorInvalidValue,
                       tr("Presence is not valid."));
        return QtTapioca::PresenceState();
    }

    // Check the account that the presence state will be changed for actually
    // exists. If it doesn't, send an error reply and return.
    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return QtTapioca::PresenceState();
    }

    // If the account is already set to the presence state requested, then
    // return successfully as there is nothing more to do.
    if (presence(account_handle) == presence_state.name())
    { return presence_state; }

    // Set the AccountConnector to store the new presence state locally.
    QVariant presenceStateVariant;
    presenceStateVariant.setValue(presence_state);
    d->m_connector->setValue(account_handle, Decibel::name_presence,
                             presenceStateVariant);

    // Get the current presence message from the AccountConnector, as we are not
    // going to change that, but we need to pass it to the next method.
    QVariantMap presence_params = presenceParameters(account_handle);

    // Call the method that will update the presence state and message remotely.
    qDebug() << "AccountManager::setPresence():"
             << "Finally in this method, we call the manageConnection() method to update the presence-state remotely.";
    return d->manageConnection(account_handle, presence_state, presence_params);
}

QtTapioca::PresenceState AccountManager::presence(const uint account_handle)
{
    qDebug() << "AccountManager::presence():"
             << "Calling m_connector->hasAccount().";
    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return QtTapioca::PresenceState();
    }

    QVariantMap account_data = d->getAccount(account_handle);
    QtTapioca::PresenceState result = account_data[Decibel::name_presence].value<QtTapioca::PresenceState>();

    //Q_ASSERT(!result.isEmpty());
    // FIXME: Should we check for UnsetType?

    return result;
}

QtTapioca::PresenceState AccountManager::currentPresence(const uint account_handle)
{
    if (!d->m_connector->hasAccount(account_handle))
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return QtTapioca::PresenceState();
    }

    QVariantMap account_data = d->getAccount(account_handle);
    QtTapioca::PresenceState result = account_data[Decibel::name_current_presence].value<QtTapioca::PresenceState>();

    //Q_ASSERT(!result.isEmpty());    // FIXME: PresenceState

    return result;
}
// FIXME: PresenceState return value here???
QtTapioca::PresenceState AccountManager::setPresenceMessage(const uint account_handle,
                                           const QString & message)
{
    return setPresenceAndMessage(account_handle, presence(account_handle),
                                 message);
}

QString AccountManager::presenceMessage(const uint account_handle) const
{
    QString result;

    QVariantMap parameters = presenceParameters(account_handle);
    if (!parameters.isEmpty() &&
        parameters.contains("message"))
    { result = parameters["message"].toString(); }

    return result;
}

QtTapioca::PresenceState AccountManager::setPresenceParameters(const uint account_handle,
                                              const QVariantMap & params)
{
    // FIXME: PresenceState return value here???
    return setPresenceAndParameters(account_handle, presence(account_handle),
                                    params);
}

QVariantMap AccountManager::presenceParameters(const uint account_handle) const
{
    QVariantMap account_data = d->getAccount(account_handle);
    if (!account_data.isEmpty() &&
        account_data.contains(Decibel::name_presence_parameters))
    { return account_data[Decibel::name_presence_parameters].toMap(); }
    return QVariantMap();
}

QtTapioca::PresenceState AccountManager::setPresenceAndMessage(const uint account_handle,
                                              const QtTapioca::PresenceState & presence_state,
                                              const QString & message)
{
    QVariantMap params(presenceParameters(account_handle));
    if (params.isEmpty()) { return QString(); }

    params.insert("message", message);

    return setPresenceAndParameters(account_handle, presence_state, params);
}

QtTapioca::PresenceState AccountManager::setPresenceAndParameters(const uint account_handle,
                                                 const QtTapioca::PresenceState & presence_state,
                                                 const QVariantMap & params)
{
    QVariantMap account_data = d->getAccount(account_handle);
    if (account_data.isEmpty())
    {
        sendErrorReply(Decibel::ErrorNoSuchAccount,
                       tr("Account %1 not found.", "1: account_handle").
                           arg(account_handle));
        return QString();
    }

    QtTapioca::PresenceState result;
    result = d->manageConnection(account_handle, presence_state, params);

    if (Decibel::isOffline(presence_state.name()))  // FIXME: PresenceState: check the type for offline instead
    {
        // not connected anymore, so we won't get any signals from telepathy
        // about the message change...
        QVariant presenceStateVariant;
        presenceStateVariant.setValue(presence_state);
        d->m_connector->setValue(account_handle, Decibel::name_presence,
                                 presenceStateVariant);
        d->m_connector->setValue(account_handle,
                                 Decibel::name_presence_parameters,
                                 QVariant(params));
    }
    return result;
}


QString AccountManager::serviceName(const uint account_handle) const
{
    if (d->connectionOf(account_handle) == 0) { return QString(); }
    return d->connectionOf(account_handle)->serviceName();
}

QString AccountManager::objectPath(const uint account_handle) const
{
    if (d->connectionOf(account_handle) == 0) { return QString(); }
    return d->connectionOf(account_handle)->objectPath();
}

void AccountManager::onOwnPresenceUpdated(QtTapioca::Connection * connection,
                                          const QtTapioca::PresenceState & presence_state,
                                          const QVariantMap & params)
{
    int account_handle = d->findAccountWithConnection(connection);
    if (0 == account_handle) { return; }

    Q_ASSERT(account_handle != 0);
    Q_ASSERT(d->m_connector->hasAccount(account_handle));

    d->updatePresenceLocally(account_handle, presence_state, params);
    emit accountUpdated(account_handle);
}

void AccountManager::onConnectionClosed(QtTapioca::Connection * connection)
{
    qDebug() << "AccountManager::onConnectionClosed(): called.";
    int account_handle = d->findAccountWithConnection(connection);
    if (0 == account_handle) { return; }

    Q_ASSERT(d->m_connector->hasAccount(account_handle));

    d->updatePresenceLocally(account_handle,
                            QtTapioca::PresenceState(Decibel::state_offline,
                                    QtTapioca::PresenceState::OfflineType),
                             presenceParameters(account_handle));

    d->setConnection(account_handle, 0);

    emit accountUpdated(account_handle);
}

void AccountManager::doBlockReconnects()
{ d->m_blockReconnects = true; }

void AccountManager::onConnectionOpened(QtTapioca::Connection * connection)
{
    // Get the list of supported presence states for this connection.
    QList<QtTapioca::PresenceState> presenceStates = connection->supportedPresenceStates();

    QList<QVariant> presenceStateVariants;
    foreach(QtTapioca::PresenceState presenceState, presenceStates)
    {
        QVariant var;
        var.setValue(presenceState);
        presenceStateVariants << var;
    }

    // Insert them into the Acount Connector.
    d->m_connector->setValue(d->findAccountWithConnection(connection), "decibel_supported_presence_states", presenceStateVariants);
}
