/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 * Copyright (C) 2008 George Goldberg <grundleborg@googlemail.com>
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DECIBEL_PLUGINHELPER_CONTACTCONNECTORBASE_H
#define DECIBEL_PLUGINHELPER_CONTACTCONNECTORBASE_H

#include <Decibel/decibel_pluginhelper_export.h>

#include <QtCore/QObject>
#include <QtCore/QString>
#include <QtCore/QVariantMap>

#include <QtTapioca/ContactBase>

/**
 * @brief A API for connectors to contact data management systems.
 *
 *
 * The ContactConnector is a abstract base class defining an interface
 * used by the ContactManager to interact with the users preferred PIM system.
 *
 * The URIs here used must be of the form
 * protocol://uri-as-used-by-telepathy-CM.
 *
 * @author Tobias Hunger <info@basyskom.de>
 */

class DECIBEL_PLUGINHELPER_EXPORT ContactConnectorBase : public QObject
{
    Q_OBJECT
public:
    /** @brief Constructor */
    explicit ContactConnectorBase(QObject * parent = 0);
    /** @brief Destructor. */
   virtual  ~ContactConnectorBase();

    /**
     * @brief Open contact data storage.
     */
   virtual void openStorage() = 0;
    /**
     * @brief Check whether a contact of the given ID exists.
     * @param contact_url A unique url for a contact.
     * @param cookie The channel cookie.
     * @returns True if the contact exists and false otherwise.
     *
     * Check whether a contact of the given ID exists.
     */
    virtual void gotContact(const QString & contact_url,
                            const quint64 cookie) const = 0;

    /**
     * @brief Get contact information from the pim system.
     * @param contact_url A unique url for a contact.
     * @param cookie The channel cookie.
     *
     * Get contact information from the pim system. Returns a empty
     * QVariantMap if the contact was unknown to the PIM system.
     */
    virtual void contact(const QString & contact_url, const quint64 cookie) const = 0;

    /**
     * @brief Update presence information on a contact.
     * @param contact_url A unique url for a contact.
     * @param presence_state The new presence state.
     * @param presence_params The new presence parameters.
     * @param cookie The channel cookie.
     *
     * This method updates the presence information of a contact in the
     * PIM system the connector talks to.
     */
    virtual void setPresence(const QString & contact_url,
                             const QString & presence_state,
                             const QVariantMap & presence_params,
                             const quint64 cookie) = 0;

    /**
     * @brief Find the contact that owns the URI.
     * @param uri The uri used.
     * @param cookie The channel cookie.
     * @return The contact ID that own given URI. 0 if the URI was not found.
     *
     * This method returns the ID of the contact owning the given URI.
     */
    virtual void findURI(const QString & uri, const quint64 cookie) const = 0;


    /**
     * @brief Get a list of URIs for a given combination of protocol and
     *        contact.
     * @param contact_url A unique url for a contact.
     * @param proto The protocol.
     * @param cookie The channel cookie.
     * @return A list of all URIs known for the given combination of contact
     *         and protocol.
     *
     * This method returns a list of all URIs that are known for a given
     * combination of contact ID and protocol.
     */
    virtual void getURIs(const QString & contact_url,
                         const QString & proto,
                         const quint64 cookie) const = 0;

    /**
     * @brief Adds a contact to the PIM DB the connector is associated with.
     * @param contact_infos The infos to be stored in the DB.
     * @param cookie The channel cookie.
     * @return The ID of the new contact. A ID of 0 indicates an error.
     *
     * This method adds a contact to the DB the connector is associated with.
     * It fetches its information from the IM network and adds it to the
     * database.
     */
    virtual void addContact(const QVariantMap & contact_infos,
                            const quint64 cookie) = 0;

Q_SIGNALS:
    /**
     * @brief Emitted when the results of a gotContact() request are ready.
     * @param got Indicates whether or not the contact exists in the PIM db.
     * @param cookie The request cookie.
     */
    void contactGot(bool got, quint64 cookie) const;
    /**
     * @brief Emitted when the results of a getURIs() request are ready.
     * @param uris A list of all the URIs for the given contact. Is an
     * empty QStringList if provided contact_url does not have any URIs.
     * @param cookie The request cookie.
     */
    void urisGot(const QStringList uris, quint64 cookie) const;
    /**
     * @brief Emitted when the results of a findURI() request are ready.
     * @param contact_url A unique url for a contact. A empty string indicates
     *                    the URI was not found for any contact.
     * @param cookie The request cookie.
     */
    void uriFound(const QString contact_url, quint64 cookie) const;
    /**
     * @brief Emitted when the results of a addContact() request are ready.
     * @param contact_url A unique url for a contact.
     *                    A empty string signifies that the addition failed.
     * @param cookie The request cookie.
     */
    void contactAdded(QString contact_url, quint64 cookie) const;
    /**
     * @brief Emitted when the results of a contact() request are ready.
     * @param details The details of the contact requested. Is an empty
     * QVariantMap if the contact id provided does not exist.
     * @param cookie The request cookie.
     */
    void contactDetails(const QVariantMap details, quint64 cookie) const;
    /**
     * @brief Signal contact data availability.
     * @param available Indicates whether the contact data is available or not.
     *
     * This signal is emitted after the initialization of the ContactConnector
     * to indicate whether it is initialized successfully, and if so, it is now
     * ready to receive requests for other data.
     */
    void contactDataAvailable(const bool available) const;
};

Q_DECLARE_INTERFACE(ContactConnectorBase, "org.kde.decibel.ContactConnector/1.0")

#endif // header guard
