/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2008 by George Goldberg <grundleborg@googlemail.com>
 *  @author George Goldberg <grundleborg@googlemail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "accountdataserialization.h"

#include <QtTapioca/PresenceState>

#include <QtCore/QList>
#include <QtXml/QDomDocument>

// Private functions, not exported in library.
QDomElement serializedItemNode(QDomDocument & domDocument, const QVariant & value);
QDomElement serializedItemList(QDomDocument & domDocument, const QList<QVariant> & list);
QVariant deserializedItemNode(const QDomElement & itemElement);


QByteArray
serializeAccountDataVariant(const QVariant & value)
{
    // Create a QDomDocument.
    QDomDocument domDocument("decibelaccountdataitem");

    // Create a root XML element and append it to the domDocument.
    QDomElement rootElement = domDocument.createElement("decibelaccountdataitem");
    domDocument.appendChild(rootElement);

    QDomElement itemElement;
    // Check if the value is a QList.
    if((value.type() == QVariant::List) || (value.type() == QVariant::StringList))
    {
        // It's a list, so we recurse into this function again.
        itemElement = serializedItemList(domDocument, value.toList());
    }
    else
    {
        // It's not a list, so we call the other function.
        itemElement = serializedItemNode(domDocument, value);
    }

    if(itemElement.isNull()) { return QByteArray(); }
    else { rootElement.appendChild(itemElement); }

    return domDocument.toByteArray();
}

QDomElement
serializedItemNode(QDomDocument & domDocument, const QVariant & value)
{
    // Create our Item element.
    QDomElement itemElement;

    // Go through each possible type that we support, and serialize appropriately.
    switch(value.type())
    {
        case QVariant::Bool:
        {
            itemElement = domDocument.createElement("item");
            itemElement.setAttribute("type", "bool");
            itemElement.setAttribute("value", value.toBool());
            break;
        }
        case QVariant::Int:
        {
            itemElement = domDocument.createElement("item");
            itemElement.setAttribute("type", "int");
            itemElement.setAttribute("value", value.toInt());
            break;
        }
        case QVariant::String:
        {
            itemElement = domDocument.createElement("item");
            itemElement.setAttribute("type", "string");
            itemElement.setAttribute("value", value.toString());
            break;
        }
        case QVariant::UInt:
        {
            itemElement = domDocument.createElement("item");
            itemElement.setAttribute("type", "uint");
            itemElement.setAttribute("value", value.toUInt());
            break;
        }
        default:
        {
            QtTapioca::PresenceState presenceState = value.value<QtTapioca::PresenceState>();
            if(!presenceState.name().isEmpty())
            {
                itemElement = domDocument.createElement("item");
                itemElement.setAttribute("type", "presencestate");

                QtTapioca::PresenceState presenceState = value.value<QtTapioca::PresenceState>();

                QDomElement psNameElement = domDocument.createElement("part");
                QDomElement psTypeElement = domDocument.createElement("part");
                QDomElement psMaySetOnSelfElement = domDocument.createElement("part");

                psNameElement.setAttribute("name", "name");
                psNameElement.setAttribute("value", presenceState.name());
                psTypeElement.setAttribute("name", "type");
                psTypeElement.setAttribute("value", presenceState.type());
                psMaySetOnSelfElement.setAttribute("name", "maysetonself");
                psMaySetOnSelfElement.setAttribute("value", presenceState.maySetOnSelf());

                itemElement.appendChild(psNameElement);
                itemElement.appendChild(psTypeElement);
                itemElement.appendChild(psMaySetOnSelfElement);

                break;
            }

            if(value.canConvert(QVariant::String))
            {
                itemElement = domDocument.createElement("item");
                itemElement.setAttribute("type", "string");
                itemElement.setAttribute("value", value.toString());
            }
        }
    }

    return itemElement;
}

QDomElement
serializedItemList(QDomDocument & domDocument, const QList<QVariant> & list)
{
    // Create our list element.
    QDomElement listElement = domDocument.createElement("list");

    // Loop over all the items in the list. If they are of type QList<QVariant>
    // then we recurse again into this method with them, otherwise we call
    // serializedItemNode() with them.

    foreach(const QVariant & value, list)
    {
        QDomElement itemElement;
        // Check if the value is a QList.
        if((value.type() == QVariant::List) || (value.type() == QVariant::StringList))
        {
            // It's a list, so we recurse into this function again.
             itemElement = serializedItemList(domDocument, value.toList());
        }
        else
        {
            // It's not a list, so we call the other function.
            itemElement = serializedItemNode(domDocument, value);
        }
        // Get the item element for this item.

        if(!itemElement.isNull()) { listElement.appendChild(itemElement); }
    }
    return listElement;
}

QVariant
deserializeAccountDataVariant(const QByteArray & data)
{
    // Create a QVariant which we will return.
    QVariant result;

    // Create a QDomDocument from the data passed.
    QDomDocument domDocument("decibelaccountdataitem");
    domDocument.setContent(data);

    // Get the root element of the XML document.
    QDomElement rootElement = domDocument.documentElement();

    // Look at the first child of the root element.
    QDomNode itemNode = rootElement.firstChild();

    // Try to convert the item node to an element.
    QDomElement itemElement = itemNode.toElement();

    // If the itemElement is empty, we return an invalid QVariant.
    if(itemElement.isNull()) { return QVariant(); }

    result = deserializedItemNode(itemElement);

    // Return the resulting QVariant.
    return result;
}

QVariant
deserializedItemNode(const QDomElement & itemElement)
{
    // Set up the return variable.
    QVariant result;

    // Deserialize the itemElement depending on what type it is.
    QString itemType = itemElement.attribute("type");

    if(itemElement.tagName() == "list")
    {
        // We have a list, recurse.
        QList<QVariant> variantList;

        QDomNode childNode = itemElement.firstChild();
        while(!childNode.isNull())
        {
            // Try to convert the node to an element.
            QDomElement childElement = childNode.toElement();
            if(!childElement.isNull())
            {
                variantList.append(deserializedItemNode(childElement));
            }

            childNode = childNode.nextSibling();
        }

        result.setValue(variantList);
    }
    else if(itemElement.tagName() == "item")
    {
        // We have an item, deserialize.
        if(itemType == "bool")
        {
            bool raw = itemElement.attribute("value").toInt();
            result.setValue(raw);
        }
        else if(itemType == "int")
        {
            int raw = itemElement.attribute("value").toInt();
            result.setValue(raw);
        }
        else if(itemType == "string")
        {
            QString raw = itemElement.attribute("value");
            result.setValue(raw);
        }
        else if(itemType == "uint")
        {
            uint raw = itemElement.attribute("value").toUInt();
            result.setValue(raw);
        }
        else if(itemType == "presencestate")
        {
            QString psName;
            QString psType;
            QString psMaySetOnSelf;

            // Loop over the children of itemElement and set the component vars of
            // the presence state using them.
            QDomNode n = itemElement.firstChild();
            while(!n.isNull())
            {
                QDomElement e = n.toElement();
                if(e.isNull()) { continue; }

                QString atrName = e.attribute("name");
                if(atrName == "name") { psName = e.attribute("value"); }
                else if(atrName == "type") { psType = e.attribute("value"); }
                else if(atrName == "maysetonself")
                { psMaySetOnSelf = e.attribute("value"); }

                // Advance to the next node.
                n = n.nextSibling();
            }

            // Create a PresenceState object from the deserialized parts.
            QtTapioca::PresenceState raw(psName, static_cast<QtTapioca::PresenceState::Type>(psType.toInt()), psMaySetOnSelf.toInt());
            result.setValue(raw);
        }
    }
    return result;
}
