/*
 * the Decibel Realtime Communication Framework
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "contactconnector.h"

#include <QtCore/QSettings>
#include <QtCore/QStringList>
#include <QtCore/QDebug>
#include <QtCore/QTimer>

#include <QtCore/QtPlugin>

namespace
{
    static const QString uri_key("URI");

    static const QString filename("~/.decibel_contact_data.ini");
    static const QString array_name("Contacts");
} // namespace

ContactConnector::ContactConnector(QObject *parent) :
        ContactConnectorBase(parent)
{ }

ContactConnector::~ContactConnector()
{
    QString use_filename(filename);
    if (filename.startsWith('~'))
    {
        QString home(qgetenv("HOME"));
        use_filename = home + filename.mid(1);
    }
    QSettings settings(use_filename, QSettings::IniFormat);
    settings.beginWriteArray(array_name);
    int array_index = 0;
    foreach (const QVariantMap & contact, m_contacts)
    {
        settings.setArrayIndex(array_index);
        ++array_index;
        const QStringList keys(contact.keys());
        foreach (const QString & key, keys)
        { settings.setValue(key, contact[key]); }
    }
    settings.endArray();
}

void ContactConnector::openStorage()
{
    QString use_filename(filename);
    if (filename.startsWith('~'))
    {
        QString home(qgetenv("HOME"));
        use_filename = home + filename.mid(1);
    }
    qDebug() << "SCC: Reading contacts from:" << use_filename;

    QSettings settings(use_filename, QSettings::IniFormat);
    int size = settings.beginReadArray(array_name);
    for (int i = 0; i < size; ++i) {
        settings.setArrayIndex(i);
        QVariantMap current_contact;

        const QStringList keys(settings.childKeys());
        foreach(const QString & current_key, keys)
        { current_contact[current_key] = settings.value(current_key); }

        Q_ASSERT(current_contact.contains(uri_key));

        qDebug() << "SCC:" << (i + 1) << "uri:" << current_contact[uri_key];
        m_contacts.append(current_contact);
    }

    settings.endArray();

    QTimer::singleShot(10, this, SLOT(sendContactDataAvailable()));
}

void ContactConnector::gotContact(const QString & contact_url, const quint64 cookie) const
{
    int contact_id(contact_url.toInt());
    if (0 == contact_id && m_contacts.size() < contact_id)
    {
        emit contactGot(false, cookie);
        return;
    }
    emit contactGot(true, cookie);
    return;
}

void ContactConnector::contact(const QString & contact_url, const quint64 cookie) const
{
    int contact_id(contact_url.toInt());
    if (0 == contact_id && m_contacts.size() < contact_id)
    {
        emit contactDetails(QVariantMap(), cookie);
        return;
    }
    emit contactDetails(m_contacts[contact_id - 1], cookie);
    return;
}

void ContactConnector::getURIs(const QString & contact_url,
                               const QString & proto,
                               const quint64 cookie) const
{
    QStringList result;
    int contact_id(contact_url.toInt());
    if (0 == contact_id && m_contacts.size() < contact_id)
    {
        emit urisGot(result, cookie);
        return;
    }
    Q_ASSERT(m_contacts[contact_id - 1].contains(uri_key));

    QString contact_proto = m_contacts[contact_id - 1][uri_key].toString();
    contact_proto = contact_proto.left(contact_proto.indexOf(QString("://")));
    if (contact_proto != proto)
    {
        emit urisGot(QStringList(), cookie);
        return;
    }

    result << m_contacts[contact_id - 1][uri_key].toString();
    emit urisGot(result, cookie);
    return;
}

void ContactConnector::findURI(const QString & uri,
                               const quint64 cookie) const
{
    for (int i = 0; i < m_contacts.size(); ++i)
    {
        Q_ASSERT(m_contacts[i].contains(uri_key));

        if (m_contacts[i][uri_key] == uri)
        {
            emit uriFound(QString::number(i + 1), cookie);
            return;
        }
    }
    emit uriFound(QString(), cookie);
    return;
}

void ContactConnector::addContact(const QVariantMap & new_contact, const quint64 cookie)
{
    m_contacts.append(new_contact);
    emit contactAdded(QString::number(m_contacts.size()), cookie);
    return;
}

void ContactConnector::setPresence(const QString & contact_url,
                                   const QString & presence_state,
                                   const QVariantMap & params,
                                   const quint64 cookie)
{
    Q_UNUSED(cookie);
    int contact_id(contact_url.toInt());
    if (0 == contact_id && m_contacts.size() < contact_id) { return; }

    qDebug() << m_contacts[contact_id - 1]["Name"] << "has a new presence:"
             << presence_state << params[QString("message")];
}

void
ContactConnector::sendContactDataAvailable()
{
    emit contactDataAvailable(true);
}

Q_EXPORT_PLUGIN2(contactconnector_simplistic, ContactConnector)
