/*
 * Copyright (C) 2006 by basyskom GmbH
 *  @author Tobias Hunger <info@basyskom.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "decibelchannelhandler.h"
#include "log.h"

#include <Decibel/Types>

#include <QtCore/QDebug>
#include <QtCore/QFile>
#include <QtCore/QTextStream>
#include <QtCore/QDateTime>

#include <QtTapioca/Channel>
#include <QtTapioca/Connection>
#include <QtTapioca/ChannelTarget>
#include <QtTapioca/ContactGroup>
#include <QtTapioca/Contact>
#include <QtTapioca/Handle>

class DecibelChannelHandlerPrivate
{
public:
    DecibelChannelHandlerPrivate(Log * logptr, DecibelChannelHandler * h) :
        m_log(logptr),
        m_handler(h)
    {
        Q_ASSERT(0 != m_log);
        Q_ASSERT(0 != m_handler);
    }

    ~DecibelChannelHandlerPrivate()
    { }

    void addChannel(QtTapioca::Connection * connection,
                    QtTapioca::Channel * channel,
                    const bool incoming)
    {
        ChannelData data;
        Decibel::LogEntry * entry = new Decibel::LogEntry();
        data.channel = channel;
        data.logEntry = entry;
        entry->protocol = connection->protocol();
        entry->startTime = QDateTime::currentDateTime().toTime_t();
        entry->endTime = 0;
        entry->incoming = incoming;

        // Find contacts:
        QStringList contacts;
        if (channel->hasGroupSupport())
        {
            qDebug() << "Channel has group support.";
            QtTapioca::ContactGroup * group = channel->contactGroup();
            if (0 == group)
            {
                qDebug() << "ContactGroup is NULL!";
                contacts.append("unknown");
                entry->connectTime = entry->startTime;
            }
            else
            {
                qDebug() << "GOT CONTACTGROUP!";
                entry->connectTime = 0;

                QObject::connect(group,
                                 SIGNAL(contactEntered(QtTapioca::Contact *)),
                                 m_handler,
                                 SLOT(doNewMember()));

                QtTapioca::UserContact * self_contact =
                    channel->connection()->userContact();
                const QList<QtTapioca::Contact *> contact_ptrs(group->contacts());
                foreach (QtTapioca::Contact * current_contact_ptr, contact_ptrs)
                {
                    qDebug() << "Examining:" << current_contact_ptr->uri();
                    if ((*current_contact_ptr) == (*self_contact))
                    {
                        qDebug() << "Skipping self in confirmed list...";
                        continue;
                    }
                    contacts.append(current_contact_ptr->uri());
                }
                const QList<QtTapioca::Contact *> group_contact_ptrs(group->pendingContacts());
                foreach (QtTapioca::Contact * current_contact_ptr, group_contact_ptrs)
                {
                    qDebug() << "Examining:" << current_contact_ptr->uri();
                    if ((*current_contact_ptr) == (*self_contact))
                    {
                        qDebug() << "Skipping self in pending list...";
                        continue;
                    }
                    contacts.append(current_contact_ptr->uri());
                }
            }
        }
        else
        { contacts.append(channel->target()->uri()); }
        entry->contacts = contacts;

        qDebug() << "Contacts set to:" << contacts;

        channels.append(data);
    }

    void removeChannel(QtTapioca::Channel * channel)
    {
        for (int i = 0; i < channels.size(); ++i)
        {
            if (channels[i].channel == channel)
            {
                Decibel::LogEntry * entry = channels[i].logEntry;
                channels[i].logEntry = 0;
                channels.removeAt(i);

                Q_ASSERT(0 != entry);
                entry->endTime = QDateTime::currentDateTime().toTime_t();
                m_log->addEntry(entry);
                return;
            }
        }
    }

    struct ChannelData
    {
        QtTapioca::Channel * channel;
        Decibel::LogEntry * logEntry;
    };

    QList<struct ChannelData> channels;

private:
    Log * const m_log;
    DecibelChannelHandler * const m_handler;
};

// ---------------------------------------------------------------

DecibelChannelHandler::DecibelChannelHandler(Log * log, QObject * parent) :
    Decibel::ChannelHandler(parent),
    d(new DecibelChannelHandlerPrivate(log, this))
{ }

DecibelChannelHandler::~DecibelChannelHandler()
{ delete d; }

bool DecibelChannelHandler::handleChannel(QtTapioca::Connection * connection,
                                          QtTapioca::Channel * channel,
                                          const bool incoming)
{
    Q_ASSERT(connection != 0);
    Q_ASSERT(channel != 0);

    connect(channel, SIGNAL(closed()), this, SLOT(doChannelClosed()));

    d->addChannel(connection, channel, incoming);

    return false;
}

void DecibelChannelHandler::doChannelClosed()
{
    QtTapioca::Channel * channel = dynamic_cast<QtTapioca::Channel *>(sender());
    if (0 == channel) { return; }
    d->removeChannel(channel);
}

void DecibelChannelHandler::doNewMember()
{
    qDebug() << "doNewMember...";
    QtTapioca::ContactGroup * group =
        dynamic_cast<QtTapioca::ContactGroup *>(sender());
    QtTapioca::Channel * channel = group->channel();

    // find entry:
    Decibel::LogEntry * entry(0);
    for (int i = 0; i < d->channels.size(); ++i)
    {
        if (d->channels[i].channel == channel)
        {
            entry = d->channels[i].logEntry;
            Q_ASSERT(0 != entry);
        }
    }
    if (0 == entry) { return; }

    bool got_connected(false);
    QtTapioca::UserContact * self_contact =
        channel->connection()->userContact();

    const QList<QtTapioca::Contact *> contact_ptrs(group->contacts());
    if (contact_ptrs.size() > 1) { got_connected = true; }
    QString uri;
    foreach (QtTapioca::Contact * current_contact_ptr, contact_ptrs)
    {
        qDebug() << "Examining:" << current_contact_ptr->uri();
        if ((*current_contact_ptr) == (*self_contact))
        {
            qDebug() << "Skipping self in confirmed list...";
            continue;
        }
        uri = current_contact_ptr->uri();
        if (!entry->contacts.contains(uri))
        { entry->contacts.append(uri); }
    }

    // Set connectTime (if unset):
    if (entry->connectTime == 0 && got_connected)
    { entry->connectTime = QDateTime::currentDateTime().toTime_t(); }
}

#include "decibelchannelhandler.moc"
